package ru.yandex.webmaster3.storage.recommendedquery.dao;

import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.storage.clickhouse.ClickhouseTable;
import ru.yandex.webmaster3.storage.recommendedquery.RecommendedQueryInfoUtil;
import ru.yandex.webmaster3.storage.recommendedquery.filtres.RecommendedQueryField;
import ru.yandex.webmaster3.storage.recommendedquery.samples.RecommendedQuery;
import ru.yandex.webmaster3.storage.util.clickhouse2.AbstractClickhouseDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseException;
import ru.yandex.webmaster3.storage.util.clickhouse2.condition.Condition;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.*;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.cases.Case;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by ifilippov5 on 10.03.17.
 */
public class RecommendedQueriesCHDao extends AbstractClickhouseDao {
    private static final Logger log = LoggerFactory.getLogger(RecommendedQueriesCHDao.class);

    private RecommendedQueriesTables recommendedQueriesTables;

    private static Where filter(Where st, Condition filters) {
        if (filters == null) {
            return st;
        }
        return st.and(new Case() {
            @Override
            public String toString() {
                return filters.toQuery();
            }
        });
    }

    public long getQueriesCount(WebmasterHostId hostId, Condition filters, Include include) {
        ClickhouseTable tableInfo = recommendedQueriesTables.tableForInclude(include);
        Where st = QueryBuilder.select()
                .countAll()
                .from(tableInfo.getLocalTableName())
                .where(QueryBuilder.eq(F.HOST_ID.getName(), hostId));

        log.debug("Get queries list: {}", st);

        st = filter(st, filters);

        log.debug("Get queries list with filter: {}", st);
        return getClickhouseServer().queryOne(tableInfo.chContext(getClickhouseServer(), hostId),
                st.toString(), r -> r.getLongUnsafe(0)).orElse(0L);
    }

    // TODO оптимизировать на пачку хостов
    public boolean exist(WebmasterHostId hostId, Include include) {
        ClickhouseTable tableInfo = recommendedQueriesTables.tableForInclude(include);
        Statement st = QueryBuilder.select("any(host_id) AS hostId")
                .from(tableInfo.getLocalTableName())
                .where(QueryBuilder.eq(F.HOST_ID.getName(), hostId));
        String anyHostId = getClickhouseServer().queryOne(tableInfo.chContext(getClickhouseServer(), hostId),
                st.toString(), r -> r.getString("hostId")).orElse(null);
        return (anyHostId.equals(hostId.toString()));
    }

    public List<RecommendedQuery> getQueries(
            WebmasterHostId hostId, Condition filters, RecommendedQueryField orderBy,
            OrderBy.Direction orderDirection, long limitFrom, long limitSize, Include include) throws ClickhouseException {
        ClickhouseTable tableInfo = recommendedQueriesTables.tableForInclude(include);
        Where st = QueryBuilder.select(
                F.QUERY.getName(),
                F.REGION_ID.getName(),
                F.FORECASTED_BID.getName(),
                F.FORECASTED_BUDGET.getName(),
                F.FORECASTED_CLICKS.getName(),
                F.FORECASTED_SHOWS.getName(),
                F.URL.getName(),
                F.POSITION.getName(),
                F.IS_EXTENDED.getName())
                .from(tableInfo.getLocalTableName())
                .where(QueryBuilder.eq(F.HOST_ID.getName(), hostId));

        st = filter(st, filters);
        GroupableLimitableOrderable fst = st;
        List<Pair<Object, OrderBy.Direction>> order = new ArrayList<>();
        if (orderBy != null) {
            order.add(Pair.of(F.fromRecommendedQueryField(orderBy).getName(), orderDirection));
        }
        order.add(Pair.of(F.fromRecommendedQueryField(RecommendedQueryField.WEIGHT).getName(), OrderBy.Direction.DESC));
        if (orderBy != RecommendedQueryField.QUERY) {
            order.add(Pair.of(F.fromRecommendedQueryField(RecommendedQueryField.QUERY).getName(), OrderBy.Direction.ASC));
        }
        fst = fst.orderBy(order)
                 .limit((int) limitFrom, (int) limitSize);
        return getClickhouseServer().queryAll(tableInfo.chContext(getClickhouseServer(), hostId), fst.toString(),
                chRow -> new RecommendedQuery(
                        hostId,
                        chRow.getString(F.QUERY.name),
                        chRow.getLong(F.REGION_ID.name),
                        RecommendedQueryInfoUtil.BID_VALUE_CONVERTER.apply(chRow.getDouble(F.FORECASTED_BID.name)),
                        chRow.getDouble(F.FORECASTED_BUDGET.name),
                        chRow.getDouble(F.FORECASTED_CLICKS.name),
                        chRow.getDouble(F.FORECASTED_SHOWS.name),
                        chRow.getString(F.URL.name),
                        chRow.getInt(F.POSITION.name),
                        chRow.getInt(F.IS_EXTENDED.name) > 0
                )
        );
    }

    public long getRegionsCount(WebmasterHostId hostId, Include include) throws ClickhouseException {
        ClickhouseTable tableInfo = recommendedQueriesTables.tableForInclude(include);
        Where st = QueryBuilder.selectDistinct(F.REGION_ID.getName())
                .countAll()
                .from(tableInfo.getLocalTableName())
                .where(QueryBuilder.eq(F.HOST_ID.getName(), hostId));

        return getClickhouseServer().queryOne(tableInfo.chContext(getClickhouseServer(), hostId),
                st.toString(), r -> r.getLongUnsafe(0)).orElse(0L);
    }

    public List<Long> getRegions(WebmasterHostId hostId, long limitFrom, long limitSize, Include include) throws ClickhouseException {
        ClickhouseTable tableInfo = recommendedQueriesTables.tableForInclude(include);
        GroupableLimitableOrderable st = QueryBuilder.select(F.REGION_ID.getName())
                .from(tableInfo.getLocalTableName())
                .where(QueryBuilder.eq(F.HOST_ID.getName(), hostId))
                .groupBy(F.REGION_ID.getName());

        GroupableLimitableOrderable fst = st.orderBy("count(" + F.REGION_ID.getName() + ")", OrderBy.Direction.DESC)
                .limit((int) limitFrom, (int) limitSize);

        return getClickhouseServer().queryAll(tableInfo.chContext(getClickhouseServer(), hostId), fst.toString(),
                chRow -> chRow.getLong(F.REGION_ID.name));
    }

    @Required
    public void setRecommendedQueriesTables(RecommendedQueriesTables recommendedQueriesTables) {
        this.recommendedQueriesTables = recommendedQueriesTables;
    }

    public enum Include {
        REGULAR, EXTENDED, BOTH
    }

    public enum F {
        HOST_ID("host_id"),
        IS_EXTENDED("is_extended"),
        QUERY("query"),
        REGION_ID("region_id"),
        FORECASTED_BID("forecasted_bid"),
        FORECASTED_BUDGET("forecasted_budget"),
        FORECASTED_CLICKS("forecasted_clicks"),
        FORECASTED_SHOWS("forecasted_shows"),
        URL("url"),
        POSITION("position"),
        WEIGHT("weight");

        private final String name;

        F(String name) {
            this.name = name;
        }

        private static F fromRecommendedQueryField(RecommendedQueryField value) {
            switch (value) {
                case QUERY:
                    return QUERY;
                case FORECASTED_BID:
                    return FORECASTED_BID;
                case FORECASTED_BUDGET:
                    return FORECASTED_BUDGET;
                case FORECASTED_CLICKS:
                    return FORECASTED_CLICKS;
                case FORECASTED_SHOWS:
                    return FORECASTED_SHOWS;
                case URL:
                    return URL;
                case POSITION:
                    return POSITION;
                case WEIGHT:
                    return WEIGHT;
                case REGION_ID:
                    return REGION_ID;
                case IS_EXTENDED:
                    return IS_EXTENDED;
            }
            throw new IllegalStateException();
        }

        public String getName() {
            return name;
        }
    }
}
