package ru.yandex.webmaster3.storage.recommendedquery.dao;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.LocalDate;
import org.springframework.stereotype.Repository;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.ValueDataMapper;

import java.util.Collection;
import java.util.List;
import java.util.function.Consumer;

@Slf4j
@Repository
public class RecommendedQueriesLastVisitDateYDao extends AbstractYDao {

    public RecommendedQueriesLastVisitDateYDao() {
        super(PREFIX_QUERIES, "recommended_queries_last_visit_date2");
    }

    public long count() {
        return countAll().queryOne();
    }

    public void insert(WebmasterHostId hostId, LocalDate lastVisitDate) {
        upsert(
            F.HOST_ID.value(hostId),
            F.LAST_VISIT_DATE.value(lastVisitDate)
        ).execute();
    }

    public void forEachHost(Consumer<Pair<WebmasterHostId, LocalDate>> consumer) {
        streamReader(MAPPER, consumer);
    }

    public LocalDate getLastVisitDate(WebmasterHostId hostId) {
        return select(F.LAST_VISIT_DATE)
                .where(F.HOST_ID.eq(hostId))
                .queryOne();
    }

    public List<Pair<WebmasterHostId, LocalDate>> getLastVisitDate(Collection<WebmasterHostId> hostIds) {
        return select(MAPPER)
                .where(F.HOST_ID.in(hostIds))
                .queryForList();
    }

    public void batchInsert(List<Pair<WebmasterHostId, LocalDate>> items) {
        batchInsert(VALUE_MAPPER, items).execute();
    }

    private static final DataMapper<Pair<WebmasterHostId, LocalDate>> MAPPER = DataMapper.create(F.HOST_ID, F.LAST_VISIT_DATE, Pair::of);
    private static final ValueDataMapper<Pair<WebmasterHostId, LocalDate>> VALUE_MAPPER = ValueDataMapper.create2(
            Pair.of(F.HOST_ID, Pair::getKey),
            Pair.of(F.LAST_VISIT_DATE, Pair::getValue)
    );

    private static final class F {
        public static final Field<WebmasterHostId> HOST_ID = Fields.hostIdField("host_id");
        public static final Field<LocalDate> LAST_VISIT_DATE = Fields.jodaDateField("last_visit_date");
    }
}
