package ru.yandex.webmaster3.storage.searchbase.dao;

import java.util.List;
import java.util.NavigableMap;
import java.util.function.Function;

import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.Instant;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.searchbase.SearchBaseUpdateInfo;
import ru.yandex.webmaster3.core.util.W3Collectors;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;

/**
 * @author avhaliullin
 */
@Repository
public class SearchBaseUpdatesYDao extends AbstractYDao {

    private static final Pair<Field<Instant>, Function<SearchBaseUpdateInfo, Instant>> KEY_FOR_SORT = Pair.of(F.SWITCH_DATE, SearchBaseUpdateInfo::getBaseSwitchDate);

    public SearchBaseUpdatesYDao() {
        super(PREFIX_WEBMASTER3, "search_base_updates");
    }

    public void insertBaseUpdateInfo(SearchBaseUpdateInfo baseUpdateInfo) {
        upsert(
                F.SWITCH_DATE.value(baseUpdateInfo.getBaseSwitchDate()),
                F.COLLECTION_DATE.value(baseUpdateInfo.getBaseCollectionDate())
        ).execute();
    }

    public NavigableMap<Instant, SearchBaseUpdateInfo> getUpdatesHistory() {
        return select(MAPPER)
                .queryForList(KEY_FOR_SORT)
                .stream()
                .collect(
                        W3Collectors.toTreeMap(
                                SearchBaseUpdateInfo::getBaseSwitchDate,
                                Function.identity()
                        )
                );
    }

    public List<SearchBaseUpdateInfo> getUpdatesHistoryList() {
        return select(MAPPER)
                .queryForList(KEY_FOR_SORT);
    }

    public static final DataMapper<SearchBaseUpdateInfo> MAPPER = DataMapper.create(
            F.SWITCH_DATE, F.COLLECTION_DATE, SearchBaseUpdateInfo::new
    );

    private static class F {
        static final Field<Instant> SWITCH_DATE = Fields.jodaInstantField("switch_date");
        static final Field<Instant> COLLECTION_DATE = Fields.jodaInstantField("collection_date");
    }
}
