package ru.yandex.webmaster3.storage.searchquery;

import com.google.common.collect.Range;
import com.google.common.collect.RangeMap;
import com.google.common.collect.RangeSet;
import com.google.common.collect.TreeRangeMap;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.LocalDate;
import ru.yandex.webmaster3.core.searchquery.QueryIndicator;
import ru.yandex.webmaster3.storage.searchquery.util.Accumulator;
import ru.yandex.webmaster3.storage.searchquery.util.ExtractorFactory;

import java.util.*;
import java.util.function.Function;
import java.util.function.UnaryOperator;
import java.util.stream.Collectors;

/**
 * @author aherman
 */
public class AccumulatorMap implements Function<DayStat, Void> {
    private final RangeMap<LocalDate, Map<QueryIndicator, Accumulator>> rangeAccumulators;

    private AccumulatorMap(RangeMap<LocalDate, Map<QueryIndicator, Accumulator>> rangeAccumulators) {
        this.rangeAccumulators = rangeAccumulators;
    }

    @Override
    public Void apply(DayStat stat) {
        Map<QueryIndicator, Accumulator> accMap = rangeAccumulators.get(stat.getDate());
        if (accMap == null) {
            return null;
        }
        for (Accumulator accumulator : accMap.values()) {
            accumulator.apply(stat);
        }
        return null;
    }

    public List<Pair<Range<LocalDate>, Double>> getIndicator(QueryIndicator indicator) {
        return getIndicatorNormalized(indicator, null);
    }

    /**
     * Находит первые ненулевые значения (для нормировки)
     * @param indicator
     * @param skip
     * @return
     */
    public Map<QueryIndicator, Double> findFirstNotZeroValue(Collection<QueryIndicator> indicators, int skip) {
        Map<QueryIndicator, Double> result = new EnumMap<>(QueryIndicator.class);
        int index = 0;
        boolean multipleIndicators = indicators.size() > 1;
        for (Map.Entry<Range<LocalDate>, Map<QueryIndicator, Accumulator>> entry : rangeAccumulators
                .asMapOfRanges().entrySet()) {
            if (index++ < skip) {
                continue;
            }
            // ищем айтем со всеми нужными значениями, не равными null
            boolean valid = true;
            for (QueryIndicator indicator : indicators) {
                Accumulator accumulator = entry.getValue().get(indicator);
                if (accumulator == null || accumulator.getValue() == null || accumulator.getValue() == 0D ||
                        (accumulator.hasGaps() && multipleIndicators)) {
                    valid = false;
                    break;
                }
                result.put(indicator, accumulator.getValue());
            }
            if (valid) {
                return result;
            }
        }
        // костыль, если все значения по одному параметры равны 0 (или их нет)
        if (multipleIndicators) {
            for (QueryIndicator indicator : indicators) {
                result.putAll(findFirstNotZeroValue(Collections.singletonList(indicator), skip));
            }
        }
        return result;
    }

    public List<Pair<Range<LocalDate>, Double>> getIndicatorNormalized(QueryIndicator indicator, Double normFactor) {
        List<Pair<Range<LocalDate>, Double>> result = new ArrayList<>();
        for (Map.Entry<Range<LocalDate>, Map<QueryIndicator, Accumulator>> entry : rangeAccumulators
                .asMapOfRanges().entrySet()) {
            Accumulator accumulator = entry.getValue().get(indicator);
            Double value = null;
            if (accumulator != null) {
                value = accumulator.getValue();
            }
            if (value != null && normFactor != null) {
                value /= normFactor;
            }
            result.add(Pair.of(entry.getKey(), value));
        }
        return result;
    }

    public static AccumulatorMap create(List<QueryIndicator> indicators, RangeSet<LocalDate> ranges) {
        RangeMap<LocalDate, Map<QueryIndicator, Accumulator>> accumulators = TreeRangeMap.create();
        for (Range<LocalDate> range : ranges.asRanges()) {
            Map<QueryIndicator, Accumulator> accMap = indicators.stream()
                    .collect(Collectors.toMap(UnaryOperator.identity(), ExtractorFactory::createExtractor));
            accumulators.put(range, accMap);
        }
        return new AccumulatorMap(accumulators);
    }
}
