package ru.yandex.webmaster3.storage.searchquery;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import lombok.AllArgsConstructor;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.searchquery.OrderDirection;
import ru.yandex.webmaster3.core.searchquery.Path;
import ru.yandex.webmaster3.core.searchquery.PathId;
import ru.yandex.webmaster3.core.searchquery.QueryIndicator;
import ru.yandex.webmaster3.core.searchquery.SpecialGroup;
import ru.yandex.webmaster3.storage.searchquery.dao.TopUrlsTextsCHDao;
import ru.yandex.webmaster3.storage.searchquery.dao.TopUrlsValuesCHDao;

/**
 * @author lester
 */
@AllArgsConstructor(onConstructor_ = @Autowired)
@Service("urlStatisticsService2")
public class UrlStatisticsService2 {

    private final TopUrlsTextsCHDao mdbTopUrlsTextsCHDao;
    private final TopUrlsValuesCHDao mdbTopUrlsValuesCHDao;

    public List<PathStat> getStatistics(SpecialGroup specialGroup, WebmasterHostId hostId,
                                        List<QueryIndicator> indicators, List<PathId> pathIds,
                                        Set<Integer> regions, RegionInclusion regionInclusion,
                                        LocalDate dateFrom, LocalDate dateTo, DeviceType deviceType) {
        if (pathIds.isEmpty()) {
            return Collections.emptyList();
        }
        switch (specialGroup) {
            case TOP_URLS:
                return mdbTopUrlsValuesCHDao.getPathStat(
                        hostId,
                        indicators,
                        dateFrom, dateTo,
                        regionInclusion, regions,
                        pathIds, deviceType);
            default:
                return Collections.emptyList();
        }
    }

    public Map<PathId, String> getPathTexts(WebmasterHostId hostId, SpecialGroup specialGroup, List<PathId> pathIds) {
        if (pathIds.isEmpty()) {
            return Collections.emptyMap();
        }
        switch (specialGroup) {
            case TOP_URLS:
                Map<PathId, String> idToPathText;
                idToPathText = mdbTopUrlsTextsCHDao.getPaths(hostId, pathIds)
                        .stream()
                        .collect(Collectors.toMap(Pair::getKey, Pair::getValue));
                return idToPathText;
            default:
                return Collections.emptyMap();
        }
    }

    public List<PathId> getPathIds(WebmasterHostId hostId, SpecialGroup specialGroup,
                                   Set<Integer> regions, RegionInclusion regionInclusion,
                                   LocalDate userRangeStart, LocalDate userRangeEnd,
                                   List<PathId> excludePaths, DeviceType deviceType,
                                   QueryIndicator orderBy, OrderDirection orderDirection,
                                   int startFrom, int limit) {
        switch (specialGroup) {
            case TOP_URLS:
                return mdbTopUrlsValuesCHDao.getPathIds(
                        hostId,
                        userRangeStart, userRangeEnd,
                        excludePaths,
                        regionInclusion, regions,
                        deviceType,
                        orderBy, orderDirection,
                        startFrom, limit);
            default:
                return Collections.emptyList();
        }
    }

    public int countPaths(WebmasterHostId hostId, SpecialGroup specialGroup,
                          Set<Integer> regions, RegionInclusion regionInclusion,
                          LocalDate userRangeStart, LocalDate userRangeEnd, DeviceType deviceType) {
        switch (specialGroup) {
            case TOP_URLS:
                return mdbTopUrlsValuesCHDao.countQueries(
                        hostId,
                        userRangeStart, userRangeEnd,
                        regionInclusion, regions,
                        deviceType
                );
            default:
                return 0;
        }
    }

    public Pair<Map<PathId, String>, List<PathStat>> getPathStatistics(WebmasterHostId hostId,
                                                                       SpecialGroup specialGroup,
                                                                       List<QueryIndicator> indicators,
                                                                       RegionInclusion regionInclusion, Set<Integer> regions,
                                                                       List<PathId> pathIds,
                                                                       LocalDate rangeStart, LocalDate rangeEnd, DeviceType deviceType) {
        Map<PathId, String> idToPathText;
        List<PathStat> pathStat;
        switch (specialGroup) {
            case TOP_URLS:
                idToPathText = mdbTopUrlsTextsCHDao.getPaths(hostId, pathIds).stream()
                        .collect(Collectors.toMap(Pair::getKey, Pair::getValue));

                pathStat = mdbTopUrlsValuesCHDao.getPathStat(
                        hostId,
                        indicators,
                        rangeStart, rangeEnd,
                        regionInclusion, regions,
                        pathIds, deviceType);
                break;

            default:
                idToPathText = Collections.emptyMap();
                pathStat = Collections.emptyList();
        }
        return Pair.of(idToPathText, pathStat);
    }

    public List<Path> suggestPath(WebmasterHostId hostId, SpecialGroup specialGroup,
                                  String pathFilter,
                                  int pageSize) {
        switch (specialGroup) {
            case TOP_URLS:
                return mdbTopUrlsTextsCHDao.suggestQueries(hostId, pathFilter, pageSize);
            default:
                return Collections.emptyList();
        }
    }
}
