package ru.yandex.webmaster3.storage.searchquery.dao;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import lombok.Setter;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.searchquery.LatestSearchQuery;
import ru.yandex.webmaster3.core.searchquery.OrderDirection;
import ru.yandex.webmaster3.core.searchquery.QueryFilter;
import ru.yandex.webmaster3.core.searchquery.QueryId;
import ru.yandex.webmaster3.core.searchquery.QueryIndicator;
import ru.yandex.webmaster3.storage.clickhouse.LocalClickhouseTableProvider;
import ru.yandex.webmaster3.storage.clickhouse.TableProvider;
import ru.yandex.webmaster3.storage.clickhouse.TableType;
import ru.yandex.webmaster3.storage.clickhouse.table.QueriesWeekSmallTable;
import ru.yandex.webmaster3.storage.util.clickhouse2.AbstractClickhouseDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHRow;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseEscapeUtils;

/**
 * @author aherman
 */
public abstract class AbstractQueriesWeekCHDao extends AbstractClickhouseDao {

    private static final int WEEK_LENGTH = 7;
    @Setter
    private TableProvider tableStorage;

    protected LocalClickhouseTableProvider getTableProvider() {
        return new QueriesWeekSmallTable(tableStorage.getTable(TableType.WEEK_QUERIES_SMALL));
    }

    private static double fromWeekToNormal(long value) {
        return (double) value / WEEK_LENGTH;
    }

    private static float fromValueToFilter(float value) {
        return value * WEEK_LENGTH;
    }

    public Map<QueryId, LatestSearchQuery> getQueriesByIds(WebmasterHostId hostId, List<QueryId> queryIds) {
        LocalClickhouseTableProvider table = getTableProvider();
        if (queryIds.isEmpty()) {
            return Collections.emptyMap();
        }
        String queriesFilter = createFilteringQuery(
                table.getTableName(getClickhouseServer(), hostId), hostId, Collections.emptyList(), null, null, queryIds, true);
        String query = "SELECT * FROM (" + queriesFilter + ")";

        List<LatestSearchQuery> queries = getClickhouseServer().queryAll(
                table.chContext(getClickhouseServer(), hostId), query, AbstractQueriesWeekCHDao::chRowToQueryWithQueryId);

        return queries.stream().collect(Collectors.toMap(LatestSearchQuery::getQueryId, q -> q));
    }

    @NotNull
    static LatestSearchQuery chRowToQueryWithQueryId(CHRow r) {
        QueryId queryId = new QueryId(r.getLongUnsafe(F.QUERY_ID));
        double showsCount = fromWeekToNormal(r.getLongUnsafe(F.ST));
        double clicksCount = fromWeekToNormal(r.getLongUnsafe(F.CT));
        double ctr = r.getDouble(F.CTR);
        double avgShowPosition = r.getDouble(F.POS1_50);
        double avgClickPosition = r.getDouble(F.POC1_50);
        return new LatestSearchQuery(queryId, null, showsCount, clicksCount, ctr, avgShowPosition, avgClickPosition);
    }

    @NotNull
    static LatestSearchQuery chRowToQueryWithUrl(CHRow r) {
        String url = r.getString(F.URL);
        double showsCount = fromWeekToNormal(r.getLongUnsafe(F.ST));
        double clicksCount = fromWeekToNormal(r.getLongUnsafe(F.CT));
        double ctr = r.getDouble(F.CTR);
        double avgShowPosition = r.getDouble(F.POS1_50);
        double avgClickPosition = r.getDouble(F.POC1_50);
        return new LatestSearchQuery(null, url, showsCount, clicksCount, ctr, avgShowPosition, avgClickPosition);
    }

    // TODO временная костыльная функция на время переезда, нужна из-за изменения типа колонки Int64 -> UInt64
    protected String getQueryIdsString(List<QueryId> queryIds) {
        return queryIds.stream().map(q -> Long.toString(q.getQueryId())).collect(Collectors.joining(","));
    }

    public Map<QueryId, String> getQueryText(WebmasterHostId hostId, List<QueryId> queryIds) {
        if (queryIds.isEmpty()) {
            return Collections.emptyMap();
        }
        LocalClickhouseTableProvider table = getTableProvider();
        String q = "SELECT DISTINCT " + F.QUERY_ID +", " + F.QUERY_TEXT
                + " FROM " + table.getTableName(getClickhouseServer(), hostId)
                + " PREWHERE host_id = '" + hostId.toStringId() + "' AND " + F.QUERY_ID +" IN (" + getQueryIdsString(queryIds) + ")";
        return getClickhouseServer().queryAll(table.chContext(getClickhouseServer(), hostId), q, r -> {
            QueryId queryId = new QueryId(r.getLongUnsafe("query_id"));
            String queryText = r.getString("query_text");
            return Pair.of(queryId, queryText);
        }).stream().collect(Collectors.toMap(Pair::getKey, Pair::getValue));
    }

    protected String toCond(List<QueryFilter> filters) {
        List<String> parts = new ArrayList<>(filters.size());
        for (QueryFilter filter : filters) {
            String f = null;
            float filterValue = filter.getNumberValue();

            switch(filter.getIndicator()) {
                case TOTAL_SHOWS_COUNT: f = F.ST; filterValue = fromValueToFilter(filterValue); break;
                case TOTAL_CLICKS_COUNT: f = F.CT; filterValue = fromValueToFilter(filterValue); break;
                case TOTAL_CTR: f = F.CTR; break;
                case SHOWS_COUNT_1: f = F.S1; filterValue = fromValueToFilter(filterValue); break;
                case CLICKS_COUNT_1: f = F.C1; filterValue = fromValueToFilter(filterValue); break;
                case SHOWS_COUNT_2_3: f = F.S2_3; filterValue = fromValueToFilter(filterValue); break;
                case CLICKS_COUNT_2_3: f = F.C2_3; filterValue = fromValueToFilter(filterValue); break;
                case SHOWS_COUNT_4_10: f = F.S4_10; filterValue = fromValueToFilter(filterValue); break;
                case CLICKS_COUNT_4_10: f = F.C4_10; filterValue = fromValueToFilter(filterValue); break;
                case SHOWS_COUNT_11_50: f = F.S11_50; filterValue = fromValueToFilter(filterValue); break;
                case CLICKS_COUNT_11_50: f = F.C11_50; filterValue = fromValueToFilter(filterValue); break;
                case CTR_1: f = F.CTR1; break;
                case CTR_2_3: f = F.CTR2_3; break;
                case CTR_4_10: f = F.CTR4_10; break;
                case CTR_11_50: f = F.CTR11_50; break;
                case AVERAGE_SHOW_POSITION: f = F.POS1_50; break;
                case AVERAGE_CLICK_POSITION: f = F.POC1_50; break;
                case AVERAGE_SHOW_POSITION_2_3: f = F.POS2_3; break;
                case AVERAGE_CLICK_POSITION_2_3: f = F.POC2_3; break;
                case AVERAGE_SHOW_POSITION_4_10: f = F.POS4_10; break;
                case AVERAGE_CLICK_POSITION_4_10: f = F.POC4_10; break;
                case AVERAGE_SHOW_POSITION_11_50: f = F.POS11_50; break;
                case AVERAGE_CLICK_POSITION_11_50: f = F.POC11_50; break;
            }
            if (f != null) {
                switch (filter.getOperation()) {
                    case GREATER_THAN: f += " > "; break;
                    case GREATER_EQUAL: f += " >= "; break;
                    case LESS_THAN: f += " < "; break;
                    case LESS_EQUAL: f += " <= "; break;
                    default: continue;
                }
                f += Float.toString(filterValue);
                parts.add(f);
            }
        }

        return String.join(" AND ", parts);
    }

    @NotNull
    protected String getQueryTextCondition(List<QueryFilter> filters) {
        List<String> textConditions = new ArrayList<>();
        for (QueryFilter textFilter : filters) {
            if (textFilter.getIndicator() != QueryIndicator.QUERY) {
                continue;
            }
            String text = textFilter.getText();
            if (!StringUtils.isEmpty(text)) {
                String q = Stream.of(StringUtils.split(text, '|'))
                        .map(StringUtils::trimToEmpty)
                        .filter(p -> !StringUtils.isEmpty(p))
                        .map(p -> F.QUERY_TEXT + " LIKE '%" + ClickhouseEscapeUtils.trimAndEscape(p) + "%'")
                        .collect(Collectors.joining(" OR "));

                if (!StringUtils.isEmpty(q)) {
                    textConditions.add("(" + q + ")");
                }
            }
        }
        if (textConditions.isEmpty()) {
            return "1 = 1";
        }
        return String.join(" AND ", textConditions);
    }

    String getUrlCondition(List<QueryFilter> filters) {
        StringBuilder result = new StringBuilder(1024);
        for (QueryFilter filter : filters) {
            if (filter.getIndicator() == QueryIndicator.URL) {
                String text = ClickhouseEscapeUtils.trimAndEscape(filter.getText());
                if (!StringUtils.isEmpty(text)) {
                    if (result.length() > 0) {
                        result.append(" AND ");
                    }
                    result.append(F.URL).append(" LIKE '%").append(text).append("%'");
                }
            }
        }
        return result.toString();
    }

    String getOrder2(QueryIndicator orderBy, OrderDirection direction) {
        String o = F.ST;
        switch (orderBy) {
            case TOTAL_SHOWS_COUNT: o = F.ST; break;
            case TOTAL_CLICKS_COUNT: o = F.CT; break;
            case TOTAL_CTR: o = F.CTR; break;
            case AVERAGE_SHOW_POSITION: o = F.POS1_50; break;
            case AVERAGE_CLICK_POSITION: o = F.POC1_50; break;
        }
        if (direction == OrderDirection.DESC) {
            o += " DESC";
        } else {
            o += " ASC";
        }
        return o;
    }

    @NotNull
    protected String createFilteringQuery(String table, WebmasterHostId hostId, List<QueryFilter> filters, String order,
                                String limit, List<QueryId> queryIds, boolean forQuery) {
        String whereCondition = "1=1";

        if (!queryIds.isEmpty()) {
            whereCondition = F.QUERY_ID
                    + " IN ("
                    + queryIds.stream().map(qi -> Long.toString(qi.getQueryId())).collect(Collectors.joining(","))
                    + ")";
        } else {
            String queryTextCondition = getQueryTextCondition(filters);
            String urlCondition = getUrlCondition(filters);

            if (!StringUtils.isEmpty(queryTextCondition)) {
                whereCondition = queryTextCondition;
                if (!StringUtils.isEmpty(urlCondition)) {
                    whereCondition += " AND " + urlCondition;
                }
            } else if (!StringUtils.isEmpty(urlCondition)) {
                whereCondition = urlCondition;
            }
        }

        String parametricCondition = toCond(filters);
        if (StringUtils.isEmpty(parametricCondition)) {
            parametricCondition = "1=1";
        }

        String limitClause = "";
        if (!StringUtils.isEmpty(limit)) {
            limitClause = " LIMIT " + limit;
        }
        String orderClause = "";
        if (!StringUtils.isEmpty(order)) {
            orderClause = " ORDER BY " + order;
        }

        String query =
                "SELECT "
                + "   host_id"
                + (forQuery ? " , query_id" : " , url")
                //+ " , sum(shows_total) as " + F.ST
                //+ " , sum(clicks_total) as " + F.CT
                + " , sum(shows_1) as " + F.S1
                + " , sum(clicks_1) as " + F.C1
                + " , sum(shows_2_3) as " + F.S2_3
                + " , sum(clicks_2_3) as " + F.C2_3
                + " , sum(agr_pos_show_2_3) as APS2_3"
                + " , sum(agr_pos_click_2_3) as APC2_3"
                + " , sum(shows_4_10) as " + F.S4_10
                + " , sum(clicks_4_10) as " + F.C4_10
                + " , sum(agr_pos_show_4_10) as APS4_10"
                + " , sum(agr_pos_click_4_10) as APC4_10"
                + " , sum(shows_11_50) as " + F.S11_50
                + " , sum(clicks_11_50) as " + F.C11_50
                + " , sum(agr_pos_show_11_50) as APS11_50"
                + " , sum(agr_pos_click_11_50) as APC11_50"
                + " , (S1 + S2_3 + S4_10 + S11_50) as S1_50"
                + " , (C1 + C2_3 + C4_10 + C11_50) as C1_50"
                + " , (S1_50) as " + F.ST
                + " , (C1_50) as " + F.CT
                + " , (S1 + APS2_3 + APS4_10 + APS11_50) as APS1_50"
                + " , (C1 + APC2_3 + APC4_10 + APC11_50) as APC1_50"
                + " , (S2_3 > 0 ? APS2_3 / S2_3 : 51) as " + F.POS2_3
                + " , (C2_3 > 0 ? APC2_3 / C2_3 : 51) as " + F.POC2_3
                + " , (S4_10 > 0 ? APS4_10 / S4_10 : 51) as " + F.POS4_10
                + " , (C4_10 > 0 ? APC4_10 / C4_10 : 51) as " + F.POC4_10
                + " , (S11_50 > 0 ? APS11_50 / S11_50 : 51) as " + F.POS11_50
                + " , (C11_50 > 0 ? APC11_50 / C11_50 : 51) as " + F.POC11_50
                + " , (S1_50 > 0 ? APS1_50 / S1_50 : 51) as " + F.POS1_50
                + " , (C1_50 > 0 ? APC1_50 / C1_50 : 51) as " + F.POC1_50
                + " , (ST > 0 ? CT / ST : CT * 1.0) as " + F.CTR
                + " , (S1 > 0 ? C1 / S1 : C1 * 1.0) as " + F.CTR1
                + " , (S2_3 > 0 ? C2_3 / S2_3 : C2_3 * 1.0) as " + F.CTR2_3
                + " , (S4_10 > 0 ? C4_10 / S4_10 : C4_10 * 1.0) as " + F.CTR4_10
                + " , (S11_50 > 0 ? C11_50 / S11_50 : C11_50 * 1.0) as " + F.CTR11_50
                + " FROM " + table + " "
                + " PREWHERE host_id = '" + hostId + "'"
                + " WHERE " + whereCondition
                + " GROUP BY host_id, " + (forQuery ? "query_id" : "url")
                + " HAVING " + parametricCondition
                + " " + orderClause
                + " " + limitClause
                ;

        return query;
    }

    protected static final class F {
        public static final String HOST_ID = "host_id";
        public static final String QUERY_ID = "query_id";
        public static final String QUERY_TEXT = "query_text";
        public static final String URL      = "url";
        public static final String SHOWS_TOTAL = "shows_total";
        public static final String CLICKS_TOTAL = "clicks_total";
        public static final String SHOWS_1 = "shows_1";
        public static final String CLICKS_1 = "clicks_1";
        public static final String SHOWS_2_3 = "shows_2_3";
        public static final String CLICKS_2_3 = "clicks_2_3";
        public static final String AGR_POS_SHOW_2_3 = "agr_pos_show_2_3";
        public static final String AGR_POS_CLICK_2_3 = "agr_pos_click_2_3";
        public static final String SHOWS_4_10 = "shows_4_10";
        public static final String CLICKS_4_10 = "clicks_4_10";
        public static final String AGR_POS_SHOW_4_10 = "agr_pos_show_4_10";
        public static final String AGR_POS_CLICK_4_10 = "agr_pos_click_4_10";
        public static final String SHOWS_11_50 = "shows_11_50";
        public static final String CLICKS_11_50 = "clicks_11_50";
        public static final String AGR_POS_SHOW_11_50 = "agr_pos_show_11_50";
        public static final String AGR_POS_CLICK_11_50 = "agr_pos_click_11_50";

        public static final String ST       = "ST";
        public static final String CT       = "CT";
        public static final String S1       = "S1";
        public static final String S2_3     = "S2_3";
        public static final String S4_10    = "S4_10";
        public static final String S11_50   = "S11_50";
        public static final String C1       = "C1";
        public static final String C2_3     = "C2_3";
        public static final String C4_10    = "C4_10";
        public static final String C11_50   = "C11_50";
        public static final String POS1_50  = "POS1_50";
        public static final String POS2_3   = "POS2_3";
        public static final String POS4_10  = "POS4_10";
        public static final String POS11_50 = "POS11_50";
        public static final String POC1_50  = "POC1_50";
        public static final String POC2_3   = "POC2_3";
        public static final String POC4_10  = "POC4_10";
        public static final String POC11_50 = "POC11_50";
        public static final String CTR      = "CTR";
        public static final String CTR1     = "CTR1";
        public static final String CTR2_3   = "CTR2_3";
        public static final String CTR4_10  = "CTR4_10";
        public static final String CTR11_50 = "CTR11_50";

        public static final String HAS_URL  = "HAS_URL";
    }
}
