package ru.yandex.webmaster3.storage.searchquery.dao;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import lombok.Setter;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.joda.time.Days;
import org.joda.time.LocalDate;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.searchquery.IndicatorUsage;
import ru.yandex.webmaster3.core.searchquery.OrderDirection;
import ru.yandex.webmaster3.core.searchquery.QueryId;
import ru.yandex.webmaster3.core.searchquery.QueryIndicator;
import ru.yandex.webmaster3.storage.clickhouse.LocalClickhouseTableProvider;
import ru.yandex.webmaster3.storage.clickhouse.TableProvider;
import ru.yandex.webmaster3.storage.clickhouse.TableType;
import ru.yandex.webmaster3.storage.searchquery.DeviceType;
import ru.yandex.webmaster3.storage.searchquery.QueryStat;
import ru.yandex.webmaster3.storage.searchquery.RegionInclusion;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseException;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseQueryContext;

/**
 * @author aherman
 */
public class FavoriteValuesCHDao extends AbstractQueriesStatsCHDao {

    @Setter
    private TableProvider tableStorage;

    protected LocalClickhouseTableProvider getTableProvider() {
        return tableStorage.getTable(TableType.FAVORITES_VALUES);
    }

    public List<QueryId> getQueryIds(
            WebmasterHostId hostId,
            Set<QueryId> queryIds,
            LocalDate dateFrom, LocalDate dateTo,
            RegionInclusion regionInclusion, Set<Integer> regionIds,
            QueryIndicator orderBy, OrderDirection direction,
            int offset, int limit, DeviceType deviceType)
    {
        LocalClickhouseTableProvider table = getTableProvider();
        // похоже что проще сделать пагинацию прямо в java, нежели в clickhouse
        String regionFilter = queryStatisticsFiltersService.getRegionFilter(regionInclusion, regionIds);
        if (!regionFilter.isEmpty()) {
            regionFilter = "AND " + regionFilter;
        }
        if (!orderBy.isUsable(IndicatorUsage.IN_ORDER)) {
            orderBy = QueryIndicator.TOTAL_SHOWS_COUNT;
        }
        if (orderBy == QueryIndicator.NAME) {
            orderBy = QueryIndicator.TOTAL_SHOWS_COUNT;
        }
        String currentRangeCondition = " date >= '" + dateFrom.toString() + "'" +
                " AND date <= '" + dateTo.toString() + "' ";
        String orderColumns = getOrderColumns(orderBy, currentRangeCondition);
        String orderColumnName = getOrderColumnName(orderBy);
        String orderByClause = "";
        if (!StringUtils.isEmpty(orderColumnName) && !StringUtils.isEmpty(orderColumns)) {
            orderByClause = " ORDER BY " + orderColumnName + " " +
                    (isPosition(orderBy) ? direction.reverse().name() : direction.name());
        }

        LocalDate whereDateFrom = orderBy.isDynamics()
                ? dateFrom.minusDays(Days.daysBetween(dateFrom, dateTo).getDays() + 1)
                : dateFrom;
        String tableName = table.getTableName();
        ClickhouseQueryContext.Builder chContext = table.chContext(getClickhouseServer(), hostId);

        // main subquery
        StringBuilder queryBuilder = new StringBuilder(1024);
        queryBuilder.append("SELECT query_id");
        if (!StringUtils.isEmpty(orderColumns)) {
            queryBuilder.append(", ").append(orderColumns);
        }
        queryBuilder.append(" FROM ").append(tableName);
        queryBuilder.append(" WHERE host_id = '").append(hostId.toStringId()).append("' ");
        queryBuilder.append(" AND date >= '").append(whereDateFrom.toString()).append("' ");
        queryBuilder.append(" AND date <= '").append(dateTo.toString()).append("' ");
        queryBuilder.append(regionFilter);
        queryBuilder.append(" AND ").append(deviceType.getQueryFilter());
        queryBuilder.append(" GROUP BY query_id ").append(orderByClause);
        // effective limit
        /*limit = Math.min(queryIds.size() - offset, limit);
        if (direction == OrderDirection.DESC) {
            queryBuilder.append(" LIMIT ").append(offset).append(", ").append(limit);
        }*/

        List<QueryId> sortedQueryIds = getClickhouseServer().queryAll(chContext, queryBuilder.toString(),
                row -> new QueryId(row.getLongUnsafe("query_id")));
        List<QueryId> result = new ArrayList<>();

        if (direction == OrderDirection.ASC) {
            // adding queries from beginning
            Collection<QueryId> queriesWithoutData = CollectionUtils.subtract(queryIds, sortedQueryIds);
            queriesWithoutData.stream().skip(offset).limit(limit).forEach(result::add);
            offset = Math.max(offset - queriesWithoutData.size(), 0);
            limit = Math.max(limit - result.size(), 0);
            sortedQueryIds.stream().filter(queryIds::contains).skip(offset).limit(limit).forEach(result::add);
        } else {
            // adding queries from end
            sortedQueryIds.stream().filter(queryIds::contains).skip(offset).limit(limit).forEach(result::add);
            limit = Math.max(limit - result.size(), 0);
            if (limit > 0) {
                Collection<QueryId> queriesWithoutData = CollectionUtils.subtract(queryIds, sortedQueryIds);
                queriesWithoutData.stream().limit(limit).forEach(result::add);
            }
        }
        return result;
    }

    protected String getQueryIdsString(List<QueryId> queryIds) {
        return queryIds.stream().map(l -> String.valueOf(l.getQueryId())).collect(Collectors.joining(","));
    }

    public List<QueryStat> getQueryStat(
            WebmasterHostId hostId,
            LocalDate dateFrom, LocalDate dateTo,
            RegionInclusion regionInclusion, Set<Integer> regionIds,
            List<QueryId> queryIds, List<QueryIndicator> indicators, DeviceType deviceType)
            throws ClickhouseException
    {
        LocalClickhouseTableProvider table = getTableProvider();
        return getQueryStat(table.getTableName(), table.getShard(getClickhouseServer(), hostId),
                hostId, dateFrom, dateTo, regionInclusion, regionIds, queryIds, indicators, deviceType);
    }

}
