package ru.yandex.webmaster3.storage.searchquery.dao;

import java.io.IOException;
import java.util.List;
import java.util.UUID;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.lang3.tuple.Triple;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.searchquery.QueryGroupId;
import ru.yandex.webmaster3.core.searchquery.history.QueryGroupState;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.ValueDataMapper;

@Repository
public class QueryGroupStateYDao extends AbstractYDao {
    private static final String TABLE_NAME = "query_group_state";
    static final ObjectMapper OM = new ObjectMapper()
            .setSerializationInclusion(JsonInclude.Include.NON_NULL)
            .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);

    public QueryGroupStateYDao() {
        super(PREFIX_QUERIES, TABLE_NAME);
    }

    public List<QueryGroupState> get(WebmasterHostId hostId) {
        return select(MAPPER)
                .where(F.HOST_ID.eq(hostId))
                .queryForList();
    }

    public boolean exist(QueryGroupId groupId) {
        return select(F.HOST_ID)
                .where(F.HOST_ID.eq(groupId.getHostId()))
                .and(F.GROUP_ID.eq(groupId.getGroupId()))
                .queryOne() != null;
    }

    public void add(QueryGroupId groupId, QueryGroupState queryGroupState) {
        String data = "";
        try {
            data = OM.writeValueAsString(queryGroupState);
        } catch (JsonProcessingException ex) {
            // no
        }

        upsert(F.HOST_ID.value(groupId.getHostId()),
                F.GROUP_ID.value(groupId.getGroupId()),
                F.DATA.value(data))
                .execute();
    }

    public void delete(WebmasterHostId hostId) {
        delete()
                .where(F.HOST_ID.eq(hostId))
                .execute();
    }

    public void delete(QueryGroupId groupId) {
        delete()
                .where(F.HOST_ID.eq(groupId.getHostId()))
                .and(F.GROUP_ID.eq(groupId.getGroupId()))
                .execute();
    }

    public void batchInsert(List<Triple<WebmasterHostId, UUID, String>> items) {
        batchInsert(VALUE_MAPPER, items).execute();
    }

    static final ValueDataMapper<Triple<WebmasterHostId, UUID, String>> VALUE_MAPPER = ValueDataMapper.create(
            Pair.of(F.HOST_ID, tri -> F.HOST_ID.get(tri.getLeft())),
            Pair.of(F.GROUP_ID, tri -> F.GROUP_ID.get(tri.getMiddle())),
            Pair.of(F.DATA, tri -> F.DATA.get(tri.getRight()))
    );

    private static final DataMapper<QueryGroupState> MAPPER = DataMapper.create(
           F.HOST_ID,
           F.GROUP_ID,
           F.DATA,
           (hostId, groupId, data) -> {
                QueryGroupState state = null;
                try {
                    state = OM.readValue(data, QueryGroupState.class);
                    state.setGroupId(QueryGroupId.byGroupIdStr(hostId, groupId));
                } catch (IOException ex) {
                    //String e = ex.toString();
                    //System.out.print(e);
                }
                return state;
            }
   );

    private interface F {
        Field<WebmasterHostId> HOST_ID = Fields.hostIdField("host_id");
        Field<UUID> GROUP_ID = Fields.uuidField("group_id");
        Field<String> DATA = Fields.stringField("data");
    }
}
