package ru.yandex.webmaster3.storage.searchquery.dao;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.UUID;

import com.google.common.collect.Iterables;
import com.google.protobuf.InvalidProtocolBufferException;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.Instant;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.searchquery.QueryFilter;
import ru.yandex.webmaster3.core.searchquery.QueryGroup;
import ru.yandex.webmaster3.core.searchquery.QueryGroupId;
import ru.yandex.webmaster3.core.searchquery.QueryIndicator;
import ru.yandex.webmaster3.proto.searchquery.QueryGroupCassandra;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.QueryBuilder;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.FieldMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.ValueDataMapper;

/**
 * @author aherman
 */
@Repository
public class QueryGroupYDao extends AbstractYDao {

    private static final ValueDataMapper<QueryGroup> INSERT_VALUE_MAPPER = ValueDataMapper.create2(
            Pair.of(F.HOST_ID, qg -> qg.getQueryGroupId().getHostId()),
            Pair.of(F.QUERY_GROUP_ID, qg -> qg.getQueryGroupId().getGroupId()),
            Pair.of(F.GROUP_NAME, QueryGroup::getName),
            Pair.of(F.CREATE_DATE, QueryGroup::getCreateDate),
            Pair.of(F.UPDATE_DATE, QueryGroup::getUpdateDate),
            Pair.of(F.HAS_SELECTED_QUERIES, QueryGroup::isHasQueries),
            Pair.of(F.QUERY_FILTER, qg -> serializeFilter(qg.getFilters())));

    private static DataMapper<QueryGroup> MAPPER = DataMapper.create(F.HOST_ID, F.QUERY_GROUP_ID, F.GROUP_NAME, F.CREATE_DATE, F.UPDATE_DATE,
            F.HAS_SELECTED_QUERIES, F.QUERY_FILTER,
            (hostId, queryGroupId, name, createDate, updateDate, hasSelectedQueries, filter) -> {
                List<QueryFilter> queryFilters = deserializeFilter(filter);
                hasSelectedQueries = hasSelectedQueries != null ? hasSelectedQueries : false;
                return new QueryGroup(new QueryGroupId(hostId, queryGroupId), name, hasSelectedQueries, queryFilters,
                        createDate, updateDate);
            });

    public QueryGroupYDao() {
        super(PREFIX_QUERIES, "query_group");
    }

    private static byte[] serializeFilter(List<QueryFilter> filters) {
        if (CollectionUtils.isEmpty(filters)) {
            return null;
        }
        QueryGroupCassandra.CassandraQueryFilters.Builder builder = QueryGroupCassandra.CassandraQueryFilters.newBuilder();
        for (QueryFilter filter : filters) {
            QueryGroupCassandra.CassandraQueryFilters.CassandraQueryFilter.Builder filterBuilder = builder.addFiltersBuilder();
            filterBuilder.setIndicator(filter.getIndicator().value());
            filterBuilder.setOperation(filter.getOperation().value());
            if (filter.getOperation().isTextOperation()) {
                filterBuilder.setTextValue(filter.getText());
            } else {
                filterBuilder.setNumberValue(filter.getNumberValue());
            }
        }
        return builder.build().toByteArray();
    }

    public static List<QueryFilter> deserializeFilter(byte[] bytes) {
        if (bytes == null || bytes.length == 0) {
            return Collections.emptyList();
        }
        List<QueryFilter> result = new ArrayList<>();
        try {
            QueryGroupCassandra.CassandraQueryFilters filters = QueryGroupCassandra.CassandraQueryFilters.parseFrom(bytes);
            for (QueryGroupCassandra.CassandraQueryFilters.CassandraQueryFilter filter : filters.getFiltersList()) {
                QueryIndicator indicator = QueryIndicator.R.fromValueOrUnknown(filter.getIndicator());
                if (indicator == QueryIndicator.UNKNOWN) {
                    throw new RuntimeException("Unable to read filter, unknown indicator: " + filter.getOperation());
                }

                QueryFilter.Operation op = QueryFilter.Operation.R.fromValueOrUnknown(filter.getOperation());
                if (op == QueryFilter.Operation.UNKNOWN) {
                    throw new RuntimeException("Unable to read filter, unknown operation: " + filter.getOperation());
                }

                String textValue = null;
                float numberValue = 0;
                if (op.isTextOperation()) {
                    textValue = filter.getTextValue();
                } else {
                    numberValue = filter.getNumberValue();
                }

                result.add(new QueryFilter(indicator, op, numberValue, textValue));
            }
        } catch (InvalidProtocolBufferException e) {
            throw new RuntimeException("Unable to parse filter", e);
        }
        return result;
    }

    public List<QueryGroup> getGroups(List<WebmasterHostId> hostIds) {
        var st = select(MAPPER).where(F.HOST_ID.in(hostIds))
                .order(F.HOST_ID.asc())
                .order(F.QUERY_GROUP_ID.asc());

        return queryForList(st, MAPPER, list -> {
            var last = Iterables.getLast(list);
            WebmasterHostId lastHostId = last.getQueryGroupId().getHostId();
            UUID lastGroupId = last.getGroupId();

            List<WebmasterHostId> contHostIds = hostIds.stream().filter(h -> h.compareTo(lastHostId) >= 0).toList();
            var newSt = select(MAPPER).where(F.HOST_ID.in(contHostIds))
                    .order(F.HOST_ID.asc())
                    .order(F.QUERY_GROUP_ID.asc());

            var or1 = F.HOST_ID.gt(lastHostId);

            var or2 = QueryBuilder.and(List.of(
                    F.HOST_ID.eq(lastHostId),
                    F.QUERY_GROUP_ID.gt(lastGroupId)
            ));

            return newSt.cont(QueryBuilder.or(List.of(or1, or2))).getStatement();
        });
    }

    public void batchInsert(Collection<QueryGroup> batch) {
        batchInsert(INSERT_VALUE_MAPPER, batch).execute();
    }

    public void addGroup(WebmasterHostId hostId, UUID groupUUID, String groupName, List<QueryFilter> queryFilter, Instant createDate) {
        upsert(F.HOST_ID.value(hostId),
                F.QUERY_GROUP_ID.value(groupUUID),
                F.GROUP_NAME.value(groupName),
                F.CREATE_DATE.value(createDate),
                F.UPDATE_DATE.value(createDate),
                F.HAS_SELECTED_QUERIES.value(false),
                F.QUERY_FILTER.value(serializeFilter(queryFilter))
        ).execute();
    }

    public List<QueryGroup> listGroups(WebmasterHostId hostId) {
        return select(MAPPER)
                .where(F.HOST_ID.eq(hostId))
                .queryForList(FieldMapper.create(F.QUERY_GROUP_ID, qg -> qg.getQueryGroupId().getGroupId()));
    }

    public QueryGroup getGroup(QueryGroupId queryGroupId) {
        return select(MAPPER)
                .where(F.HOST_ID.eq(queryGroupId.getHostId()))
                .and(F.QUERY_GROUP_ID.eq(queryGroupId.getGroupId()))
                .queryOne();
    }

    public int getGroupCount(WebmasterHostId hostId) {
        return countAll()
                .where(F.HOST_ID.eq(hostId))
                .queryOne().intValue();
    }

    public void deleteGroup(QueryGroupId queryGroupId) {
        delete()
                .where(F.HOST_ID.eq(queryGroupId.getHostId()))
                .and(F.QUERY_GROUP_ID.eq(queryGroupId.getGroupId()))
                .execute();
    }

    public void changeFilter(QueryGroupId queryGroupId, List<QueryFilter> queryFilter) {
        update()
                .with(F.QUERY_FILTER.set(serializeFilter(queryFilter)))
                .where(F.HOST_ID.eq(queryGroupId.getHostId()))
                .and(F.QUERY_GROUP_ID.eq(queryGroupId.getGroupId()))
                .execute();
    }

    public void changeName(QueryGroupId queryGroupId, String groupName) {
        update()
                .with(F.GROUP_NAME.set(groupName))
                .where(F.HOST_ID.eq(queryGroupId.getHostId()))
                .and(F.QUERY_GROUP_ID.eq(queryGroupId.getGroupId()))
                .execute();
    }

    private interface F {
        Field<WebmasterHostId> HOST_ID = Fields.hostIdField("host_id");
        Field<UUID> QUERY_GROUP_ID = Fields.uuidField("query_group_id");
        Field<String> GROUP_NAME = Fields.stringField("group_name").makeOptional();
        Field<Instant> CREATE_DATE = Fields.jodaInstantField("create_date").makeOptional();
        Field<Instant> UPDATE_DATE = Fields.jodaInstantField("update_date").makeOptional();
        Field<Boolean> HAS_SELECTED_QUERIES = Fields.boolField("has_selected_queries");
        Field<byte[]> QUERY_FILTER = Fields.byteArrayField("query_filter").makeOptional();
    }
}
