package ru.yandex.webmaster3.storage.searchquery.dao;

import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import lombok.Setter;
import org.apache.commons.lang3.StringUtils;
import org.joda.time.Days;
import org.joda.time.LocalDate;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.searchquery.IndicatorUsage;
import ru.yandex.webmaster3.core.searchquery.OrderDirection;
import ru.yandex.webmaster3.core.searchquery.PathId;
import ru.yandex.webmaster3.core.searchquery.QueryIndicator;
import ru.yandex.webmaster3.storage.clickhouse.LocalClickhouseTableProvider;
import ru.yandex.webmaster3.storage.clickhouse.TableProvider;
import ru.yandex.webmaster3.storage.clickhouse.TableType;
import ru.yandex.webmaster3.storage.clickhouse.table.TopUrlsValuesTable;
import ru.yandex.webmaster3.storage.searchquery.DayStat;
import ru.yandex.webmaster3.storage.searchquery.DeviceType;
import ru.yandex.webmaster3.storage.searchquery.PathStat;
import ru.yandex.webmaster3.storage.searchquery.RegionInclusion;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseQueryContext;

/**
 * @author lester
 */
public class TopUrlsValuesCHDao extends AbstractQueriesStatsCHDao {

    @Setter
    private TableProvider tableStorage;
    @Setter
    private String partSuffix;

    protected String getPathIdsString(List<PathId> pathIds) {
        return pathIds.stream().map(l -> Long.toString(l.getValue())).collect(Collectors.joining(","));
    }

    protected LocalClickhouseTableProvider getTableProvider() {
        return new TopUrlsValuesTable(tableStorage.getTable(TableType.TOP_URLS_VALUES), partSuffix);
    }

    public int countQueries(WebmasterHostId hostId, LocalDate dateFrom, LocalDate dateTo,
                            RegionInclusion regionInclusion, Set<Integer> regionIds, DeviceType deviceType) {
        LocalClickhouseTableProvider table = getTableProvider();
        String regionFilter = queryStatisticsFiltersService.getRegionFilter(regionInclusion, regionIds);
        if (!regionFilter.isEmpty()) {
            regionFilter = "AND " + regionFilter;
        }
        String q =
                "SELECT count() AS c FROM ("
                        + "SELECT path_id FROM " + table.getTableName(getClickhouseServer(), hostId)
                        + " PREWHERE host_id = '" + hostId.toStringId() + "'"
                        + " AND date >= '" + dateFrom.toString() + "'"
                        + " AND date <= '" + dateTo.toString() + "'"
                        + " " + regionFilter
                        + " AND " + deviceType.getQueryFilter()
                        + " GROUP BY path_id"
                        + ")";
        return getClickhouseServer().queryOne(table.chContext(getClickhouseServer(), hostId), q, row -> row.getLongUnsafe("c")).orElse(0L).intValue();
    }

    public List<PathId> getPathIds(WebmasterHostId hostId,
                                   LocalDate dateFrom, LocalDate dateTo,
                                   List<PathId> excludePaths,
                                   RegionInclusion regionInclusion, Set<Integer> regionIds,
                                   DeviceType deviceType,
                                   QueryIndicator orderBy, OrderDirection direction,
                                   int limitFrom, int limit) {
        LocalClickhouseTableProvider table = getTableProvider();
        String orderColumns = "";
        String orderClause = "";
        if (!orderBy.isUsable(IndicatorUsage.IN_ORDER)) {
            orderBy = QueryIndicator.TOTAL_SHOWS_COUNT;
        }
        if (orderBy == QueryIndicator.NAME) {
            orderBy = QueryIndicator.TOTAL_SHOWS_COUNT;
        }
        String currentRangeCondition = "date >= '" + dateFrom.toString() + "'"
                + " AND date <= '" + dateTo.toString() + "' ";
        orderColumns = getOrderColumns(orderBy, currentRangeCondition);
        String orderColumnName = getOrderColumnName(orderBy);
        if (!StringUtils.isEmpty(orderColumnName) && !StringUtils.isEmpty(orderColumns)) {
            if (isPosition(orderBy)) {
                direction = direction.reverse();
            }
            orderClause = "ORDER BY " + orderColumnName;
            orderColumns = " , " + orderColumns;
            if (direction == OrderDirection.ASC) {
                orderClause += " ASC";
            } else {
                orderClause += " DESC";
            }
        }

        LocalDate whereDateFrom = orderBy.isDynamics()
                ? dateFrom.minusDays(Days.daysBetween(dateFrom, dateTo).getDays() + 1)
                : dateFrom;
        String regionFilter = queryStatisticsFiltersService.getRegionFilter(regionInclusion, regionIds);
        if (!regionFilter.isEmpty()) {
            regionFilter = "AND " + regionFilter;
        }
        String excludePathString;
        if (excludePaths.isEmpty()) {
            excludePathString = "";
        } else {
            excludePathString = " AND NOT path_id IN(" + getPathIdsString(excludePaths) + ")";
        }
        String q = "SELECT path_id " + orderColumns
                + " FROM " + table.getTableName(getClickhouseServer(), hostId)
                + " WHERE host_id = '" + hostId.toStringId() + "'"
                + " AND date >= '" + whereDateFrom.toString() + "'"
                + " AND date <= '" + dateTo.toString() + "'"
                + " " + excludePathString
                + " " + regionFilter
                + " AND " + deviceType.getQueryFilter()
                + " GROUP BY path_id"
                + " " + orderClause
                + " LIMIT " + limitFrom + ", " + limit;
        return getClickhouseServer().queryAll(table.chContext(getClickhouseServer(), hostId), q, row -> new PathId(row.getLongUnsafe("path_id")));
    }

    public List<PathStat> getPathStat(WebmasterHostId hostId,
                                       List<QueryIndicator> indicators,
                                       LocalDate dateFrom, LocalDate dateTo,
                                       RegionInclusion regionInclusion, Set<Integer> regionIds,
                                       List<PathId> pathIds, DeviceType deviceType) {
        LocalClickhouseTableProvider table = getTableProvider();
        return getPathStat(table.getTableName(getClickhouseServer(), hostId), table.getShard(getClickhouseServer(), hostId),
                hostId, dateFrom, dateTo, regionInclusion, regionIds, pathIds, indicators, deviceType);
    }

    protected List<PathStat> getPathStat(String tableName, Integer shard, WebmasterHostId hostId,
                                          LocalDate dateFrom, LocalDate dateTo,
                                          RegionInclusion regionInclusion, Set<Integer> regionIds,
                                          List<PathId> pathIds, List<QueryIndicator> indicators, DeviceType deviceType) {
        if (pathIds.isEmpty()) {
            return Collections.emptyList();
        }
        String regionFilter = queryStatisticsFiltersService.getRegionFilter(regionInclusion, regionIds);
        if (!regionFilter.isEmpty()) {
            regionFilter = "AND " + regionFilter;
        }
        StringBuilder qBuilder = new StringBuilder();
        Set<String> columnSet = new HashSet<>();

        addQueryColumns(columnSet, indicators);
        qBuilder.append("SELECT "
                + "date"
                + ", path_id");

        appendAggregationColumns(qBuilder, columnSet);
        qBuilder.append(
                " FROM " + tableName
                        + " WHERE host_id = '" + hostId.toStringId() + "'"
                        + " AND date >= '" + dateFrom.toString() + "'"
                        + " AND date <= '" + dateTo.toString() + "'"
                        + " AND path_id IN (" + getPathIdsString(pathIds) + ")"
                        + " " + regionFilter
                        + " AND " + deviceType.getQueryFilter()
                        + " GROUP BY date, path_id"
        );

        ClickhouseQueryContext.Builder ctx = ClickhouseQueryContext.useDefaults();
        if (shard != null){
            ctx = ctx.setHost(getClickhouseServer().pickAliveHostOrFail(shard));
        }
        List<PathStat> result = getClickhouseServer().queryAll(ctx, qBuilder.toString(), row -> {
            LocalDate date = row.getLocalDate("date");
            long pathId = row.getLongUnsafe("path_id");
            long totalShows = 0;
            long totalClicks = 0;
            long shows_1 = 0;
            long clicks_1 = 0;
            long shows_2_3 = 0;
            long clicks_2_3 = 0;
            long shows_4_10 = 0;
            long clicks_4_10 = 0;
            long shows_11_50 = 0;
            long clicks_11_50 = 0;
            long agr_shows_2_3 = 0;
            long agr_clicks_2_3 = 0;
            long agr_shows_4_10 = 0;
            long agr_clicks_4_10 = 0;
            long agr_shows_11_50 = 0;
            long agr_clicks_11_50 = 0;
            long totalSerps = 0;

            for (String column : columnSet) {
                switch (column) {
                    case F.ST:      totalShows = row.getLongUnsafe(F.ST); break;
                    case F.CT:      totalClicks = row.getLongUnsafe(F.CT); break;
                    case F.S1:      shows_1 = row.getLongUnsafe(F.S1); break;
                    case F.C1:      clicks_1 = row.getLongUnsafe(F.C1); break;
                    case F.S2_3:    shows_2_3 = row.getLongUnsafe(F.S2_3); break;
                    case F.C2_3:    clicks_2_3 = row.getLongUnsafe(F.C2_3); break;
                    case F.S4_10:   shows_4_10 = row.getLongUnsafe(F.S4_10); break;
                    case F.C4_10:   clicks_4_10 = row.getLongUnsafe(F.C4_10); break;
                    case F.S11_50:  shows_11_50 = row.getLongUnsafe(F.S11_50); break;
                    case F.C11_50:  clicks_11_50 = row.getLongUnsafe(F.C11_50); break;

                    case F.APS2_3:  agr_shows_2_3 = row.getLongUnsafe(F.APS2_3); break;
                    case F.APC2_3:  agr_clicks_2_3 = row.getLongUnsafe(F.APC2_3); break;
                    case F.APS4_10: agr_shows_4_10 = row.getLongUnsafe(F.APS4_10); break;
                    case F.APC4_10: agr_clicks_4_10 = row.getLongUnsafe(F.APC4_10); break;
                    case F.APS11_50: agr_shows_11_50 = row.getLongUnsafe(F.APS11_50); break;
                    case F.APC11_50: agr_clicks_11_50 = row.getLongUnsafe(F.APC11_50); break;

                    case F.SRP: totalSerps = row.getLongUnsafe(F.SRP); break;
                }
            }

            return new PathStat(date, new PathId(pathId),
                    totalShows, totalClicks,
                    shows_1, clicks_1,
                    shows_2_3, clicks_2_3,
                    shows_4_10, clicks_4_10,
                    shows_11_50, clicks_11_50,
                    agr_shows_2_3, agr_clicks_2_3,
                    agr_shows_4_10, agr_clicks_4_10,
                    agr_shows_11_50, agr_clicks_11_50,
                    totalSerps
            );
        });
        result.sort(Comparator.comparing(DayStat::getDate));
        return result;
    }
}
