package ru.yandex.webmaster3.storage.searchquery.rivals;

import java.util.AbstractMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import lombok.Setter;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;

import ru.yandex.webmaster3.core.util.W3Collectors;
import ru.yandex.webmaster3.storage.clickhouse.ClickhouseTableInfo;
import ru.yandex.webmaster3.storage.clickhouse.TableProvider;
import ru.yandex.webmaster3.storage.clickhouse.TableType;
import ru.yandex.webmaster3.storage.searchquery.DeviceType;
import ru.yandex.webmaster3.storage.util.clickhouse2.AbstractClickhouseDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHPrimitiveType;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHTable;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseException;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseHost;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseQueryContext;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseServer;
import ru.yandex.webmaster3.storage.util.clickhouse2.SimpleByteArrayOutputStream;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.OrderBy;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.QueryBuilder;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.Statement;

/**
 * DAO для хранения данных по конкурентам (показы-клики)
 * Created by Oleg Bazdyrev on 08/11/2017.
 */
public class RivalsStats2CHDao extends AbstractClickhouseDao {

    public static final CHTable TABLE = CHTable.builder()
            .database(DB_WEBMASTER3_QUERIES)
            .name("rivals_stats2_%s")
            .partitionBy("toYYYYMM(" + F.DATE + ")")
            .keyField(F.DATE, CHPrimitiveType.Date)
            .keyField(F.THEMATICS, CHPrimitiveType.String)
            .keyField(F.DEVICE_TYPE, CHPrimitiveType.UInt8)
            .field(F.TYPE, CHPrimitiveType.String)
            .field(F.CTR, CHPrimitiveType.Float64)
            .field(F.SHOWS, CHPrimitiveType.Float64)
            .field(F.CLICKS, CHPrimitiveType.Float64)
            .field(F.DOMAINS, CHPrimitiveType.Int64)
            .sharded(false)
            .build();

    @Setter
    private TableProvider tableStorage;

    public Map<LocalDate, Double> getClicks(String thematics, LocalDate from, LocalDate to,
                                            DeviceType deviceType, int minDomains) throws ClickhouseException {
        return getFieldInternal(thematics, from, to, deviceType, F.CLICKS, minDomains);
    }

    public Map<LocalDate, Double> getFieldInternal(String thematics, LocalDate from, LocalDate to, DeviceType deviceType,
                                                   String field, int minDomains) throws ClickhouseException {
        ClickhouseTableInfo table = tableStorage.getTable(TableType.RIVALS_STATS2);
        String localTableName = table.getLocalTableName();
        if (localTableName.endsWith("_merge")) {
            // TODO сохранять правильную таблицу
            localTableName = localTableName.substring(0, localTableName.length() - "_merge".length());
        }
        Statement st = QueryBuilder.select(F.DATE, QueryBuilder.sum(field).toString())
                .from(localTableName)
                .prewhere(QueryBuilder.eq(F.THEMATICS, thematics))
                .and(QueryBuilder.gte(F.DATE, from))
                .and(QueryBuilder.lte(F.DATE, to))
                .and(deviceType.getQueryFilter())
                .and(QueryBuilder.gte(F.DOMAINS, minDomains))
                .groupBy(F.DATE)
                .orderBy(F.DATE, OrderBy.Direction.ASC);

        return queryAll(st.toString(), chRow ->
                new AbstractMap.SimpleEntry<>(chRow.getLocalDate(0), chRow.getDouble(1))
        ).stream().collect(W3Collectors.toHashMap());
    }

    public Optional<DateTime> getMaxDate() {
        ClickhouseTableInfo table = tableStorage.getTable(TableType.RIVALS_STATS2);
        String localTableName = table.getLocalTableName();
        if (localTableName.endsWith("_merge")) {
            // TODO сохранять правильную таблицу
            localTableName = localTableName.substring(0, localTableName.length() - "_merge".length());
        }

        final String query = String.format("SELECT max(date) as last_date FROM %s", localTableName);
        return queryOne(query, x -> x.getDate("last_date"));
    }

    public void dropTables(String timestamp) throws ClickhouseException {
        String tableName = TABLE.replicatedMergeTreeTableName(-1, timestamp);
        getClickhouseServer().executeOnAllHosts(ClickhouseQueryContext.useDefaults(),
                "DROP TABLE IF EXISTS " + TABLE.getDatabase() + "." + tableName);
    }

    public void createTable(String timestamp, ClickhouseHost clickhouseHost) throws ClickhouseException {
        ClickhouseQueryContext.Builder context = ClickhouseQueryContext.useDefaults().setHost(clickhouseHost);
        getClickhouseServer().execute(context, ClickhouseServer.QueryType.INSERT,
                TABLE.createReplicatedMergeTree(-1, timestamp),
                Optional.empty(), Optional.empty());
    }

    public void insert(List<RivalsStats2> statsList, String timestamp, ClickhouseHost clickhouseHost) throws ClickhouseException {
        ClickhouseQueryContext.Builder context = ClickhouseQueryContext.useDefaults().setHost(clickhouseHost);
        String insertSpec = TABLE.insertSpec(-1, timestamp);
        SimpleByteArrayOutputStream bs = new SimpleByteArrayOutputStream();
        for (RivalsStats2 stats : statsList) {
            bs = packRowValues(bs,
                    toClickhouseDate(stats.getDate()),
                    stats.getThematics(),
                    stats.getDeviceType().getValue(),
                    stats.getType(),
                    stats.getCtr(),
                    stats.getShows(),
                    stats.getClicks(),
                    stats.getDomains()
            );
        }
        getClickhouseServer().insert(context, insertSpec, bs.toInputStream());
    }

    private interface F {
        String DATE = "date";
        String THEMATICS = "thematics";
        String DEVICE_TYPE = "device_type";
        String TYPE = "type";
        String CTR = "ctr";
        String SHOWS = "shows";
        String CLICKS = "clicks";
        String DOMAINS = "domains";
    }

}
