package ru.yandex.webmaster3.storage.searchurl;

import lombok.RequiredArgsConstructor;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.UserContext;
import ru.yandex.webmaster3.core.sitestructure.RawSearchUrlStatusEnum;
import ru.yandex.webmaster3.core.sitestructure.SearchUrlStatusEnum;
import ru.yandex.webmaster3.core.sitestructure.SearchUrlStatusUtil;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.TimeUtils;
import ru.yandex.webmaster3.storage.abt.AbtService;
import ru.yandex.webmaster3.storage.abt.model.Experiment;
import ru.yandex.webmaster3.storage.searchurl.samples.dao.SearchUrlFreshUrlSampleCHDao;
import ru.yandex.webmaster3.storage.searchurl.samples.data.*;
import ru.yandex.webmaster3.storage.turbo.service.TurboSearchUrlsStatisticsService;
import ru.yandex.webmaster3.storage.util.clickhouse2.condition.Condition;

import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * ishalaru
 * 17.03.2021
 **/
@Slf4j
@Component
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class SearchUrlFreshSamplesService {
    private final SearchUrlFreshUrlSampleCHDao searchUrlFreshUrlSampleCHDao;
    private final AbtService abtService;
    private final TurboSearchUrlsStatisticsService turboSearchUrlsStatisticsService;

    public long getSearchUrlSamplesCount(WebmasterHostId hostId, Condition filters) {
        if (abtService.isInExperiment(UserContext.currentUserId(), Experiment.SEARCH_URL_FRESH)) {
            return searchUrlFreshUrlSampleCHDao.getNewSamplesCount(hostId, filters);
        }
        return 0;
    }

    public List<Pair<String, Long>> countGroupingByLastAccess(WebmasterHostId hostId, Condition filters) {
        return searchUrlFreshUrlSampleCHDao.countGropingByLastAccess(hostId, filters);
    }

    public List<Pair<String,Long>> countGroupingByBaseDate(WebmasterHostId hostId,Condition filters){
        return searchUrlFreshUrlSampleCHDao.countGroupingByBaseDate(hostId,filters);
    }

    public Pair<Long, Long> findSkipForFreshAndBase(List<Pair<String, Long>> fresh, List<Pair<String, Long>> base, Long offset) {
        long skipFresh = 0;
        long skipBase = 0;
        List<Rec> collect = Stream.concat(fresh.stream().map(e -> new Rec(e.getLeft(), e.getRight(), true)),
                base.stream().map(e -> new Rec(e.getLeft(), e.getRight(), false)))
                .sorted()
                .collect(Collectors.toList());
        for (Rec rec : collect) {
            long count = Math.min(offset, rec.count);
            if (rec.isFresh) {
                skipFresh += count;
            } else {
                skipBase += count;
            }
            if (offset >= rec.count){
                offset -= count;
            }else {
                break;
            }
        }
        return Pair.of(skipFresh, skipBase);
    }

    @Value
    public static class Rec implements Comparable<Rec> {
        String date;
        Long count;
        boolean isFresh;

        @Override
        public int compareTo(@NotNull SearchUrlFreshSamplesService.Rec o) {
            var val = this.date.compareTo(o.date);
            return val != 0 ? -val : -Boolean.compare(this.isFresh, o.isFresh);
        }
    }

    public List<SearchUrlSample> getSearchUrlSamples(WebmasterHostId hostId, Condition filters, long limitFrom, long limitSize) {
        if (!abtService.isInExperiment(UserContext.currentUserId(), Experiment.SEARCH_URL_FRESH)) {
            return List.of();
        }
        final TurboSearchUrlsStatisticsService.TurboSourceStatuses turboSourceStatuses = turboSearchUrlsStatisticsService.getTurboSourceStatuses(hostId);
        return searchUrlFreshUrlSampleCHDao.getNewSamples(hostId, filters, limitFrom, limitSize)
                .stream()
                .map(sample -> new SearchUrlSample(
                        sample.getPath(),
                        IdUtils.hostIdToUrl(hostId) + sample.getPath(),
                        sample.getTitle(),
                        sample.getLastAccess().toDateTime(TimeUtils.EUROPE_MOSCOW_ZONE),
                        remapStatusInfo(turboSourceStatuses, sample)))
                .collect(Collectors.toList());
    }

    public Optional<SearchUrlSample> getSearchUrlSample(WebmasterHostId hostId, String path) {
        if (!abtService.isInExperiment(UserContext.currentUserId(), Experiment.SEARCH_URL_FRESH)) {
            return Optional.empty();
        }
        final TurboSearchUrlsStatisticsService.TurboSourceStatuses turboSourceStatuses = turboSearchUrlsStatisticsService.getTurboSourceStatuses(hostId);
        return searchUrlFreshUrlSampleCHDao.getNewSample(hostId, path)
                .map(sample -> new SearchUrlSample(
                        sample.getPath(),
                        IdUtils.hostIdToUrl(hostId) + sample.getPath(),
                        sample.getTitle(),
                        sample.getLastAccess().toDateTime(TimeUtils.EUROPE_MOSCOW_ZONE),
                        remapStatusInfo(turboSourceStatuses, sample)));
    }


    public List<SearchUrlEventSample> getSearchUrlEventSamples(WebmasterHostId hostId, Condition filters, long limitFrom, long limitSize) {
        if (!abtService.isInExperiment(UserContext.currentUserId(), Experiment.SEARCH_URL_FRESH)) {
            return List.of();
        }
        final List<SearchUrlFreshUrlSampleCHDao.Row> rawSamples = searchUrlFreshUrlSampleCHDao
                .getSamplesOrderBySearchBaseDate(hostId, filters, limitFrom, limitSize);

        final TurboSearchUrlsStatisticsService.TurboSourceStatuses turboSourceStatuses = turboSearchUrlsStatisticsService.getTurboSourceStatuses(hostId);
        return rawSamples.stream().map(sample -> new SearchUrlEventSample(
                sample.getPath(),
                IdUtils.hostIdToUrl(hostId) + sample.getPath(),
                hideDownloadEvidence(sample, sample.getTitle()),
                sample.getLastAccess() == null ? null : sample.getLastAccess().toLocalDate(),
                sample.getInsetTime(),
                SearchUrlEventType.R.fromValueOrNull(sample.getActionType()),
                remapStatusInfo(turboSourceStatuses, sample))).collect(Collectors.toList());
    }

    public long getExcludedUrlSamplesCount(WebmasterHostId hostId, Condition filters) {
        if (abtService.isInExperiment(UserContext.currentUserId(), Experiment.SEARCH_URL_FRESH)) {
            return searchUrlFreshUrlSampleCHDao.getExcludedSamplesCount(hostId, filters);
        }
        return 0;
    }

    public long getEventUrlSamplesCount(WebmasterHostId hostId, Condition filters) {
        if (abtService.isInExperiment(UserContext.currentUserId(), Experiment.SEARCH_URL_FRESH)) {
            return searchUrlFreshUrlSampleCHDao.getSamplesCount(hostId, filters);
        }
        return 0;
    }

    public List<SearchUrlSample> getExcludedUrlSamples(WebmasterHostId hostId, Condition filters, long limitFrom, long limitSize) {

        if (!abtService.isInExperiment(UserContext.currentUserId(), Experiment.SEARCH_URL_FRESH)) {
            return List.of();
        }
        final TurboSearchUrlsStatisticsService.TurboSourceStatuses turboSourceStatuses = turboSearchUrlsStatisticsService.getTurboSourceStatuses(hostId);
        return searchUrlFreshUrlSampleCHDao.getExcludedSamples(hostId, filters, limitFrom, limitSize)
                .stream()
                .map(sample -> new SearchUrlSample(
                        sample.getPath(),
                        IdUtils.hostIdToUrl(hostId) + sample.getPath(),
                        sample.getTitle(),
                        new DateTime(sample.getLastAccess(), TimeUtils.EUROPE_MOSCOW_ZONE),
                        remapStatusInfo(turboSourceStatuses, sample)))
                .collect(Collectors.toList());
    }

    public Optional<SearchUrlSample> getExcludedUrlSample(WebmasterHostId hostId, String path) {
        if (!abtService.isInExperiment(UserContext.currentUserId(), Experiment.SEARCH_URL_FRESH)) {
            return Optional.empty();
        }
        final TurboSearchUrlsStatisticsService.TurboSourceStatuses turboSourceStatuses = turboSearchUrlsStatisticsService.getTurboSourceStatuses(hostId);

        return searchUrlFreshUrlSampleCHDao.getExcludeSample(hostId, path)
                .map(sample -> new SearchUrlSample(
                        sample.getPath(),
                        IdUtils.hostIdToUrl(hostId) + sample.getPath(),
                        sample.getTitle(),
                        new DateTime(sample.getLastAccess(), TimeUtils.EUROPE_MOSCOW_ZONE),
                        remapStatusInfo(turboSourceStatuses, sample)));
    }

    public List<SearchUrlEventSample> getSearchUrlEventSamplesLight(WebmasterHostId hostId, Condition filters, long limitFrom, long limitSize) {
        if (!abtService.isInExperiment(UserContext.currentUserId(), Experiment.SEARCH_URL_FRESH)) {
            return List.of();
        }

        List<RawSearchUrlEventSample> samples = searchUrlFreshUrlSampleCHDao.getSamplesLight(hostId, filters, limitFrom, limitSize);
        return samples.stream().map(sample ->
                new SearchUrlEventSample(
                        sample.getUrl(),
                        IdUtils.hostIdToUrl(hostId) + sample.getUrl(),
                        sample.getTitle(),
                        sample.getLastAccess() == null ? null : sample.getLastAccess().toLocalDate(),
                        LocalDateTime.now(),
                        sample.getEventType(),
                        null,
                        SearchUrlCanonicalStatus.FRESH_CANONICAL
                )
        ).collect(Collectors.toList());
    }


    private static UrlStatusInfo remapStatusInfo(
            TurboSearchUrlsStatisticsService.TurboSourceStatuses turboSourceStatuses,
            SearchUrlFreshUrlSampleCHDao.Row freshSampleRow) {

        if (freshSampleRow == null) {
            return new UrlStatusInfo(
                    SearchUrlStatusEnum.NOTHING_FOUND,
                    null,
                    null,
                    null,
                    null,
                    null,
                    null,
                    null,
                    null,
                    false,
                    false,
                    false,
                    false,
                    true
            );
        } else {
            long validFromMetrikaLastAccess = freshSampleRow.getValidFromMetrikaLastAccess();
            long validFromIndexNowLastAccess = freshSampleRow.getValidFromIndexNowLastAccess();
            return new UrlStatusInfo(
                    remapStatusEnum(freshSampleRow),
                    freshSampleRow.getAddTime().toDateTime(TimeUtils.EUROPE_MOSCOW_ZONE),
                    null,
                    hideDownloadEvidence(freshSampleRow, freshSampleRow.getHttpCode()),
                    freshSampleRow.getHostId().getReadableHostname(),
                    freshSampleRow.getPath(),
                    "",
                    freshSampleRow.getRelCanonicalTarget(),
                    null,
                    TurboSearchUrlsStatisticsService.isEnabledTurboPage(0, turboSourceStatuses),
                    false,
                    UrlStatusInfo.isValidFromMetrika(validFromMetrikaLastAccess, validFromIndexNowLastAccess),
                    UrlStatusInfo.isValidFromIndexNow(validFromMetrikaLastAccess, validFromIndexNowLastAccess),
                    true
            );
        }
    }

    @Nullable
    private static <T> T hideDownloadEvidence(SearchUrlFreshUrlSampleCHDao.Row statusInfo, T data) {
        return hideDownloadEvidence(RawSearchUrlStatusEnum.FRESH, data);
    }

    @Nullable
    public static <T> T hideDownloadEvidence(RawSearchUrlStatusEnum status, T data) {
        boolean shouldHide = (
                status == RawSearchUrlStatusEnum.ROBOTS_HOST_ERROR ||
                        status == RawSearchUrlStatusEnum.ROBOTS_URL_ERROR
        );
        return shouldHide ? null : data;
    }

    private static SearchUrlStatusEnum remapStatusEnum(SearchUrlFreshUrlSampleCHDao.Row status) {
        return SearchUrlStatusUtil.raw2View(RawSearchUrlStatusEnum.FRESH, true);
    }
}
