package ru.yandex.webmaster3.storage.searchurl.download;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import org.joda.time.format.DateTimeFormat;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.codes.YandexHttpStatusUtil;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.download.SearchUrlStatusUtil;
import ru.yandex.webmaster3.core.sitestructure.RawSearchUrlStatusEnum;
import ru.yandex.webmaster3.core.sitestructure.SearchUrlStatusEnum;
import ru.yandex.webmaster3.core.sitestructure.StructureNodeFilter;
import ru.yandex.webmaster3.storage.AbstractFilter;
import ru.yandex.webmaster3.storage.TextFilterUtil;
import ru.yandex.webmaster3.storage.searchurl.ExcludedUrlFilter;
import ru.yandex.webmaster3.storage.searchurl.ExcludedUrlSampleField;
import ru.yandex.webmaster3.storage.searchurl.SearchUrlSamplesService;
import ru.yandex.webmaster3.storage.searchurl.samples.dao.ExcludedUrlConditions;
import ru.yandex.webmaster3.storage.searchurl.samples.data.SearchUrlSample;
import ru.yandex.webmaster3.storage.searchurl.samples.data.UrlStatusInfo;
import ru.yandex.webmaster3.storage.util.StructureFilterService;
import ru.yandex.webmaster3.storage.util.clickhouse2.condition.Condition;
import ru.yandex.webmaster3.storage.util.clickhouse2.condition.Operator;

/**
 * Created by ifilippov5 on 13.02.17.
 */
@Service
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class DownloadExcludedUrlsService {

    private final SearchUrlSamplesService searchUrlSamplesService;
    private final StructureFilterService structureFilterService;

    public List<ExcludedUrlCsvRow> getSamples(WebmasterHostId hostId, List<ExcludedUrlFilter> parameterFilters, Long nodeId) {
        List<SearchUrlSample> samples = getInternalSamples(hostId, parameterFilters, nodeId);

        return samples.stream()
                .map(sample -> new ExcludedUrlCsvRow(
                        Optional.of(sample).map(SearchUrlSample::getLastAccess).map(date -> date.toString(DateTimeFormat.forPattern("dd.MM.yyyy"))).orElse(""),
                        Optional.ofNullable(sample.getTitle()).orElse(""),
                        sample.getUrl(),
                        Optional.ofNullable(sample.getStatusInfo()).flatMap(s -> Optional.ofNullable(s.getHttpCode())).flatMap(YandexHttpStatusUtil::yandexHttpStatus2View).orElse("0"),
                        Optional.of(sample).map(SearchUrlSample::getStatusInfo).map(UrlStatusInfo::getStatus).map(SearchUrlStatusUtil::getStatusView).orElse(""),
                        Optional.of(sample).map(SearchUrlSample::getStatusInfo).map(SearchurlReportUtil::getTarget).orElse(""),
                        Optional.of(sample).map(SearchUrlSample::getStatusInfo).map(UrlStatusInfo::isFromSitemap).map(SearchurlReportUtil::toStringYesNo).orElse("")

                ))
                .collect(Collectors.toList());
    }

    private List<SearchUrlSample> getInternalSamples(WebmasterHostId hostId, List<ExcludedUrlFilter> parameterFilters, Long nodeId) {
        List<SearchUrlSample> samples = Collections.emptyList();
        Condition condition = makeCondition(parameterFilters, false);
        Condition freshCondition = makeCondition(parameterFilters, true);
        if (nodeId != null) {
            StructureNodeFilter nodeFilter = structureFilterService.getNodeFilter(hostId, nodeId);
            if (nodeFilter == null) {
                //Такой ноды нет, так что "отфильтровали всё"
                return samples;
            }
            ExcludedUrlConditions.PathLikeConditionBuilder condBuilder = ExcludedUrlConditions.pathLikeConditionBuilder();
            for (StructureNodeFilter.Part part : nodeFilter.getFilterParts()) {
                if (part instanceof StructureNodeFilter.StringPart) {
                    condBuilder.append(((StructureNodeFilter.StringPart) part).getValue());
                } else {
                    condBuilder.appendWildcard();
                }
            }
            condition = condition.andThen(condBuilder.build());
        }
        long samplesCount = searchUrlSamplesService.getExcludedUrlSamplesCount(hostId, condition, freshCondition);
        if (samplesCount == 0) {
            return samples;
        }
        samples = searchUrlSamplesService.getExcludedUrlSamples(hostId, condition, freshCondition, 0, samplesCount);

        return samples;
    }

    private Condition makeCondition(List<? extends AbstractFilter<ExcludedUrlSampleField>> filters, boolean isFresh) {
        Condition condition = Condition.trueCondition();
        if (filters == null) {
            return condition;
        }
        for (AbstractFilter<ExcludedUrlSampleField> filter : filters) {
            Operator dateOp = Operator.fromFilterOperation(filter.getOperation());

            switch (filter.getIndicator()) {
                case LAST_ACCESS_DATE:
                    if (dateOp == null) {
                        throw filter.invalidFilterException();
                    } else {
                        condition = condition.andThen(ExcludedUrlConditions.lastAccessCondition(dateOp, filter.parseDate()));
                    }
                    break;
                case TITLE:
                    condition = condition.andThen(TextFilterUtil.getTextCondition(filter, ExcludedUrlConditions.getTitleFieldName()));
                    break;
                case URL:
                    condition = condition.andThen(TextFilterUtil.getTextCondition(filter, ExcludedUrlConditions.getUrlFieldName()));
                    break;
                case URL_STATUS:
                    if (filter.getOperation() == AbstractFilter.Operation.EQUAL) {
                        SearchUrlStatusEnum status = SearchUrlStatusEnum.R.valueOf(filter.getValue());
                        if (!isFresh) {
                            Set<RawSearchUrlStatusEnum> rawStatuses = ru.yandex.webmaster3.core.sitestructure.SearchUrlStatusUtil.view2AllRaw(status);
                            condition = condition.andThen(
                                    Condition.or(rawStatuses.stream().map(ExcludedUrlConditions::urlStatusEquals).collect(Collectors.toList()))
                            );
                        } else {
                            condition = status == SearchUrlStatusEnum.INDEXED_SEARCHABLE ? condition.andThen(Condition.trueCondition()) : condition.andThen(Condition.falseCondition());
                        }
                    } else {
                        throw filter.invalidFilterException();
                    }
                    break;
                case HTTP_CODE:
                    if (filter.getOperation() == AbstractFilter.Operation.EQUAL) {
                        try {
                            int httpCode = Integer.parseInt(filter.getValue());
                            condition = condition.andThen(ExcludedUrlConditions.httpCodeEquals(httpCode, isFresh));
                        } catch (NumberFormatException e) {
                            throw filter.invalidFilterException();
                        }
                    } else {
                        throw filter.invalidFilterException();
                    }
                    break;
                default:
                    throw new RuntimeException("Not implemented filter for indicator " + filter.getIndicator());
            }
        }
        return condition;
    }
}
