package ru.yandex.webmaster3.storage.searchurl.history.dao;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.fasterxml.jackson.core.type.TypeReference;
import lombok.Builder;
import lombok.RequiredArgsConstructor;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.sitestructure.NewSiteStructure;
import ru.yandex.webmaster3.core.sitestructure.NewSiteStructureNode;
import ru.yandex.webmaster3.core.sitestructure.RawSearchUrlStatusEnum;
import ru.yandex.webmaster3.core.sitestructure.SearchUrlStatusEnum;
import ru.yandex.webmaster3.core.sitestructure.SearchUrlStatusUtil;
import ru.yandex.webmaster3.core.turbo.model.TurboSource;
import ru.yandex.webmaster3.core.util.FNVHash;
import ru.yandex.webmaster3.core.util.W3Collectors;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.storage.host.HostIndicatorsState;
import ru.yandex.webmaster3.storage.searchurl.history.dao.SiteStructureCHDao.SiteStructureRecord.SiteStructureRecordBuilder;
import ru.yandex.webmaster3.storage.searchurl.history.data.SearchUrlStat;
import ru.yandex.webmaster3.storage.util.clickhouse2.AbstractClickhouseDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHField;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHPrimitiveType;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHRow;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHTable;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseQueryContext;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseServer;
import ru.yandex.webmaster3.storage.util.clickhouse2.MdbClickhouseServer;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.OrderBy;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.QueryBuilder;

/**
 * Created by Oleg Bazdyrev on 04/03/2021.
 */
@Slf4j
@Repository("mdbSiteStructuresCHDao")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class SiteStructureCHDao extends AbstractClickhouseDao {

    private static final TypeReference<List<List<Long>>> LIST_OF_LONG_PAIRS_REFERENCE = new TypeReference<>() {
    };

    public static final int PARTITIONS_COUNT = 16;
    public static final String FULL_TABLE_NAME = "site_structures";
    public static final String DISTRIB_TABLE_NAME = "site_structures_distrib";
    public static final CHTable TABLE = CHTable.builder()
            .database(DB_WEBMASTER3_SEARCHURLS)
            .sharded(true)
            .parts(1)
            .name("tmp_site_structures_%s")
            .partitionBy("cityHash64(" + F.HOST_ID + ") % " + PARTITIONS_COUNT)
            .keyField(F.HOST_ID, CHPrimitiveType.String)
            .keyField(F.NODE_ID, CHPrimitiveType.Int64)
            .keyField(F.TIMESTAMP, CHPrimitiveType.Int64)
            .field(F.IS_USER_NODE, CHPrimitiveType.Int8)
            .field(F.PARENT_NODE_ID, CHPrimitiveType.Int64)
            .field(F.NAME, CHPrimitiveType.String)
            .field(F.NUM_OF_DOCS, CHPrimitiveType.Int64)
            .field(F.NUM_OF_DOCS_ON_SEARCH, CHPrimitiveType.Int64)
            .field(F.NUM_OF_DOUBLES, CHPrimitiveType.Int64)
            .field(F.NUM_OF_NEW_SEARCH_DOCS, CHPrimitiveType.Int64)
            .field(F.NUM_OF_GONE_SEARCH_DOCS, CHPrimitiveType.Int64)
            .field(F.HTTPCODES, CHPrimitiveType.String)
            .field(F.URL_STATUSES, CHPrimitiveType.String)
            .field(F.TURBO_SOURCE_INFO, CHPrimitiveType.String)
            .build();

    private static final List<String> FIELDS = TABLE.getFields().stream().map(CHField::getName).collect(Collectors.toList());
    private static final List<String> SEARCH_INDICATORS_FIELDS = Arrays.asList(F.HOST_ID, F.NODE_ID, F.TIMESTAMP, F.NUM_OF_DOCS_ON_SEARCH,
            F.NUM_OF_NEW_SEARCH_DOCS, F.NUM_OF_GONE_SEARCH_DOCS);
    private static final List<String> SEARCH_STATUSES_FIELDS = Arrays.asList(F.HOST_ID, F.NODE_ID, F.TIMESTAMP, F.URL_STATUSES);

    private final MdbClickhouseServer legacyMdbClickhouseServer;

    @Override
    protected ClickhouseServer getClickhouseServer() {
        return legacyMdbClickhouseServer;
    }

    public List<SiteStructureRecord> getSiteStructures(WebmasterHostId hostId, Instant fromDate, Instant toDate, Set<Long> nodes) {
        return getSiteStructuresInternal(hostId, fromDate, toDate, nodes, FIELDS);
    }

    public List<SiteStructureRecord> getSearchIndicators(WebmasterHostId hostId, Instant fromDate, Instant toDate, Set<Long> nodes) {
        return getSiteStructuresInternal(hostId, fromDate, toDate, nodes, SEARCH_INDICATORS_FIELDS);
    }

    public List<SiteStructureRecord> getSearchStatuses(WebmasterHostId hostId, Instant fromDate, Instant toDate, Set<Long> nodes) {
        return getSiteStructuresInternal(hostId, fromDate, toDate, nodes, SEARCH_STATUSES_FIELDS);
    }

    private List<SiteStructureRecord> getSiteStructuresInternal(WebmasterHostId hostId, Instant fromDate, Instant toDate, Set<Long> nodes, Collection<String> fields) {
        var query = QueryBuilder.select(fields).from(TABLE.getDatabase(), FULL_TABLE_NAME)
                .where(QueryBuilder.eq(F.HOST_ID, hostId.toString()));
        if (!CollectionUtils.isEmpty(nodes)) {
            query = query.and(QueryBuilder.in(F.NODE_ID, nodes));
        }
        if (fromDate != null) {
            query = query.and(QueryBuilder.gte(F.TIMESTAMP, fromDate.getMillis() / 1000L));
        }
        if (toDate != null) {
            query = query.and(QueryBuilder.lte(F.TIMESTAMP, toDate.getMillis() / 1000L));
        }
        var st = query.orderBy(F.TIMESTAMP, OrderBy.Direction.DESC);
        log.info("Executing: {}", st);
        var ctx = ClickhouseQueryContext.useDefaults().setHost(getClickhouseServer().pickAliveHostOrFail(getShard(hostId)));
        return getClickhouseServer().queryAll(ctx, st.toString(), chRow -> {
            var builder = SiteStructureRecord.builder();
            for (String field : fields) {
                MAPPERS.get(field).apply(chRow, builder);
            }
            return builder.build();
        });
    }

    public Map<WebmasterHostId, SearchUrlStat> getSearchUrlStats(Collection<WebmasterHostId> hostIds, Instant baseDate) {
        var st = QueryBuilder.select(F.HOST_ID, F.TIMESTAMP, F.NUM_OF_DOCS_ON_SEARCH, F.NUM_OF_NEW_SEARCH_DOCS, F.NUM_OF_GONE_SEARCH_DOCS)
                .from(DB_WEBMASTER3_SEARCHURLS, DISTRIB_TABLE_NAME)
                .where(QueryBuilder.in(F.HOST_ID, hostIds))
                .and(QueryBuilder.eq(F.NODE_ID, NewSiteStructure.ROOT_NODE_ID))
                .and(QueryBuilder.eq(F.TIMESTAMP, baseDate.getMillis() / 1000L));
        return getClickhouseServer().queryAll(st.toString(), chRow ->
                new SearchUrlStat(chRow.getHostId(F.HOST_ID), new Instant(chRow.getLong(F.TIMESTAMP) * 1000L), chRow.getLong(F.NUM_OF_DOCS_ON_SEARCH),
                chRow.getLong(F.NUM_OF_NEW_SEARCH_DOCS), chRow.getLong(F.NUM_OF_GONE_SEARCH_DOCS))
        ).stream().collect(Collectors.toMap(SearchUrlStat::getHostId, Function.identity(), W3Collectors.replacingMerger()));
    }

    private static final Map<String, BuilderMapper<SiteStructureRecordBuilder, ?>> MAPPERS = Stream.of(
            BuilderMapper.of(F.HOST_ID, CHRow::getHostId, SiteStructureRecordBuilder::hostId),
            BuilderMapper.of(F.NODE_ID, CHRow::getLong, SiteStructureRecordBuilder::nodeId),
            BuilderMapper.of(F.TIMESTAMP, (r, n) -> new Instant(r.getLong(n) * 1000L), SiteStructureRecordBuilder::timestamp),
            BuilderMapper.of(F.IS_USER_NODE, (r, n) -> r.getInt(n) > 0, SiteStructureRecordBuilder::isUserNode),
            BuilderMapper.of(F.PARENT_NODE_ID, (r, n) -> zeroToNull(r.getLong(n)), SiteStructureRecordBuilder::parentNodeId),
            BuilderMapper.of(F.NAME, CHRow::getString, SiteStructureRecordBuilder::name),
            BuilderMapper.of(F.NUM_OF_DOCS, CHRow::getLong, SiteStructureRecordBuilder::numOfDocs),
            BuilderMapper.of(F.NUM_OF_DOCS_ON_SEARCH, CHRow::getLong, SiteStructureRecordBuilder::numOfDocsOnSearch),
            BuilderMapper.of(F.NUM_OF_DOUBLES, CHRow::getLong, SiteStructureRecordBuilder::numOfDoubles),
            BuilderMapper.of(F.NUM_OF_NEW_SEARCH_DOCS, CHRow::getLong, SiteStructureRecordBuilder::numOfNewSearchDocs),
            BuilderMapper.of(F.NUM_OF_GONE_SEARCH_DOCS, CHRow::getLong, SiteStructureRecordBuilder::numOfGoneSearchDocs),
            BuilderMapper.of(F.HTTPCODES, counterMapFromPairs(Function.identity()), SiteStructureRecordBuilder::httpCodes),
            BuilderMapper.of(F.URL_STATUSES, counterMapFromPairs(RawSearchUrlStatusEnum.R::fromValue), SiteStructureRecordBuilder::excludedStatuses),
            BuilderMapper.of(F.TURBO_SOURCE_INFO, counterMapFromPairs(TurboSource.R::fromValue), SiteStructureRecordBuilder::searchTurboDocs)
    ).collect(Collectors.toUnmodifiableMap(BuilderMapper::getName, Function.identity()));

    private static Long zeroToNull(Long val) {
        return val == 0L ? null : val;
    }

    private static <K> BiFunction<CHRow, String, Map<K, Long>> counterMapFromPairs(Function<Integer, K> f) {
        return (r, n) -> counterMapFromPairs(r.getString(n), f);
    }

    private static <K> Map<K, Long> counterMapFromPairs(String pairsString, Function<Integer, K> f) {
        List<List<Long>> pairs = JsonMapping.readValue(pairsString, LIST_OF_LONG_PAIRS_REFERENCE);
        Map<K, Long> result = new HashMap<>();
        for (List<Long> pair : pairs) {
            result.put(f.apply(pair.get(0).intValue()), pair.get(1));
        }
        return result;
    }

    @Value
    @Builder(toBuilder = true)
    public static class SiteStructureRecord {
        WebmasterHostId hostId;
        long nodeId;
        Instant timestamp; // aka base date
        boolean isUserNode;
        Long parentNodeId;
        String name;
        long numOfDocs;
        long numOfDocsOnSearch;
        long numOfDoubles;
        long numOfNewSearchDocs;
        long numOfGoneSearchDocs;
        Map<Integer, Long> httpCodes;
        Map<RawSearchUrlStatusEnum, Long> excludedStatuses;
        Map<TurboSource, Long> searchTurboDocs;

        public NewSiteStructureNode toNewSiteStructureNode() {
            return NewSiteStructureNode.builder()
                    .nodeId(nodeId)
                    .parentNodeId(parentNodeId)
                    .name(name)
                    .docCount(numOfDocs)
                    .searchDocCount(numOfDocsOnSearch)
                    .httpCodes(httpCodes)
                    .newSearchUrls(numOfNewSearchDocs)
                    .goneSearchUrls(numOfGoneSearchDocs)
                    .excludedStatuses(excludedStatuses.entrySet().stream()
                            .map(entry -> Pair.of(SearchUrlStatusUtil.raw2View(entry.getKey(), false), entry.getValue()))
                            .collect(W3Collectors.toEnumMap(SearchUrlStatusEnum.class, Long::sum)))
                    .searchTurboDocs(searchTurboDocs)
                    .build();
        }

        public HostIndicatorsState toHostIndicatorsState() {
            return new HostIndicatorsState(numOfDocs, numOfDocsOnSearch, excludedStatuses.values().stream().mapToLong(Long::longValue).sum(), 0L, 0L);
        }
    }

    private int getShard(WebmasterHostId hostId) {
        int shards = getClickhouseServer().getShardsCount();
        return (int) FNVHash.hash64Mod(hostId.toString(), shards);
    }

    private interface F {
        String HOST_ID = "host_id";
        String NODE_ID = "node_id";
        String TIMESTAMP = "timestamp";
        String IS_USER_NODE = "is_user_node";
        String PARENT_NODE_ID = "parent_node_id";
        String NAME = "name";
        String NUM_OF_DOCS = "num_of_docs";
        String NUM_OF_DOCS_ON_SEARCH = "num_of_docs_on_search";
        String NUM_OF_DOUBLES = "num_of_doubles";
        String NUM_OF_NEW_SEARCH_DOCS = "num_of_new_search_docs";
        String NUM_OF_GONE_SEARCH_DOCS = "num_of_gone_search_docs";
        String HTTPCODES = "httpcodes";
        String URL_STATUSES = "url_statuses";
        String TURBO_SOURCE_INFO = "turbo_source_info";
    }

}
