package ru.yandex.webmaster3.storage.searchurl.offline.dao;

import java.util.List;
import java.util.UUID;

import org.joda.time.DateTime;
import org.joda.time.Instant;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.storage.searchurl.offline.data.SearchBaseImportInfo;
import ru.yandex.webmaster3.storage.searchurl.offline.data.SearchBaseImportStageEnum;
import ru.yandex.webmaster3.storage.searchurl.offline.data.SearchBaseImportTaskType;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;
import ru.yandex.webmaster3.storage.util.yt.YtPath;

/**
 * @author iceflame
 */
@Repository
public class SearchBaseImportTablesYDao extends AbstractYDao implements SearchBaseImportTablesRepository {
    private static final DataMapper<SearchBaseImportInfo> MAPPER = DataMapper.create(
            F.TASK_TYPE, F.SEARCH_BASE_DATE, F.DB_NAME, F.STAGE, F.READ_TABLE_NAME, F.SHARDS_COUNT, F.PARTS_COUNT,
            F.CLICKHOUSE_DC, F.IMPORT_TASK_IDS, F.PAUSED, F.SOURCE_TABLE, F.IMPORT_MANAGER_TASK_ID,
            F.REPLICATION_MANAGER_TASK_ID, SearchBaseImportInfo::new
    );

    public SearchBaseImportTablesYDao() {
        super(PREFIX_IMPORTER, "search_base_import_tables");
    }

    public void insertStage(SearchBaseImportInfo info) {
        upsert(
                F.TASK_TYPE.value(info.getTaskType()),
                F.SEARCH_BASE_DATE.value(info.getSearchBaseDate()),
                F.DB_NAME.value(info.getDbName()),
                F.READ_TABLE_NAME.value(info.getReadTableName()),
                F.STAGE.value(info.getStage()),
                F.LAST_UPDATE.value(DateTime.now()),
                F.SHARDS_COUNT.value(info.getShardsCount()),
                F.PARTS_COUNT.value(info.getPartsCount()),
                F.CLICKHOUSE_DC.value(info.getClickhouseDc()),
                F.IMPORT_TASK_IDS.value(info.getYtImportTaskIds()),
                F.SOURCE_TABLE.value(info.getSourceTable()),
                F.IMPORT_MANAGER_TASK_ID.value(info.getImportManagerTask()),
                F.REPLICATION_MANAGER_TASK_ID.value(info.getReplicationManagerTask())
        ).execute();
    }

    public void updateTaskPaused(SearchBaseImportTaskType taskType, Instant baseDate, boolean paused) {
        update(F.PAUSED.set(paused))
                .where(F.TASK_TYPE.eq(taskType))
                .and(F.SEARCH_BASE_DATE.eq(baseDate))
                .execute();
    }

    public List<SearchBaseImportInfo> listAllTasks() {
        // TODO pagination/or filter
        return select(MAPPER).queryForList();
    }

    public List<SearchBaseImportInfo> listBases(SearchBaseImportTaskType taskType) {
        return select(MAPPER)
                .where(F.TASK_TYPE.eq(taskType)).queryForList();
    }

    public SearchBaseImportInfo getBaseInfo(SearchBaseImportTaskType taskType, Instant baseDate) {
        return select(MAPPER)
                .where(F.TASK_TYPE.eq(taskType))
                .and(F.SEARCH_BASE_DATE.eq(baseDate))
                .queryOne();
    }

    public SearchBaseImportInfo getLatestBaseInfo(SearchBaseImportTaskType taskType) {
        return select(MAPPER)
                .where(F.TASK_TYPE.eq(taskType))
                .order(F.SEARCH_BASE_DATE.desc())
                .limit(1)
                .queryOne();
    }

    private interface F {
        Field<SearchBaseImportTaskType> TASK_TYPE = Fields.stringEnumField("task_type", SearchBaseImportTaskType.R);
        Field<Instant> SEARCH_BASE_DATE = Fields.jodaInstantField("search_base_date");
        Field<String> DB_NAME = Fields.stringField("db_name").makeOptional();
        Field<String> READ_TABLE_NAME = Fields.stringField("read_table_name").makeOptional();
        Field<SearchBaseImportStageEnum> STAGE = Fields.stringEnumField("stage", SearchBaseImportStageEnum.R).makeOptional();
        Field<DateTime> LAST_UPDATE = Fields.jodaDateTimeField("last_update").makeOptional();
        Field<Integer> SHARDS_COUNT = Fields.intField("shards_count").makeOptional();
        Field<Integer> PARTS_COUNT = Fields.intField("parts_count").makeOptional();
        Field<String> CLICKHOUSE_DC = Fields.stringField("clickhouse_dc").makeOptional();
        Field<List<UUID>> IMPORT_TASK_IDS = Fields.jsonField2("import_task_ids", Fields.UUID_LIST_REFERENCE).makeOptional();
        Field<Boolean> PAUSED = Fields.boolField("paused").withDefault(false).makeOptional();
        Field<YtPath> SOURCE_TABLE = Fields.ytPathField("source_table").makeOptional();
        Field<UUID> IMPORT_MANAGER_TASK_ID = Fields.uuidField("import_manager_task_id").makeOptional();
        Field<UUID> REPLICATION_MANAGER_TASK_ID = Fields.uuidField("replication_manager_task_id").makeOptional();
    }
}
