package ru.yandex.webmaster3.storage.searchurl.samples.dao;

import java.util.List;

import lombok.Setter;
import org.jetbrains.annotations.NotNull;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.storage.clickhouse.TableProvider;
import ru.yandex.webmaster3.storage.clickhouse.TableType;
import ru.yandex.webmaster3.storage.clickhouse.table.SearchUrlsTable;
import ru.yandex.webmaster3.storage.searchurl.samples.data.MemorandumSample;
import ru.yandex.webmaster3.storage.util.clickhouse2.AbstractClickhouseDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.condition.Condition;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.GroupableLimitableOrderable;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.OrderBy;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.QueryBuilder;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.Where;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.cases.Case;

/**
 * @author avhaliullin
 */
public class MemorandumSamplesCHDao extends AbstractClickhouseDao {

    @Setter
    private TableProvider tableStorage;

    public long getSamplesCount(WebmasterHostId hostId, Condition filters) {
        var table = tableStorage.getTable(TableType.MEMORANDUM_SAMPLES);
        Where st = QueryBuilder.select()
                .countAll()
                .from(table.getLocalTableName())
                .where(QueryBuilder.eq(F.HOST_ID, hostId));

        st = filter(st, filters);

        return getClickhouseServer().queryOne(
                table.chContext(getClickhouseServer(), hostId),
                st.toString(),
                r -> r.getLongUnsafe(0)
        ).orElse(0L);
    }

    public List<MemorandumSample> getSamples(WebmasterHostId hostId, Condition filters, long limitFrom, long limitSize) {
        var table = tableStorage.getTable(TableType.MEMORANDUM_SAMPLES);
        Where st = QueryBuilder.select(F.HOST_ID, F.PATH, F.COPYRIGHT_OBJECT, F.COPYRIGHT_OWNER)
                .from(table.getLocalTableName())
                .where(QueryBuilder.eq(F.HOST_ID, hostId));

        st = filter(st, filters);
        GroupableLimitableOrderable fst = st
                .orderBy(F.PATH, OrderBy.Direction.ASC)
                .limit((int) limitFrom, (int) limitSize);
        return getClickhouseServer().queryAll(
                table.chContext(getClickhouseServer(), hostId),
                fst.toString(),
                chRow ->
                        new MemorandumSample(
                                chRow.getString(F.PATH),
                                IdUtils.hostIdToUrl(hostId) + chRow.getString(F.PATH),
                                chRow.getString(F.COPYRIGHT_OBJECT),
                                chRow.getString(F.COPYRIGHT_OWNER)
                        ));
    }


    private static Where filter(Where st, Condition filters) {
        if (filters == null) {
            return st;
        }
        return st.and(new Case() {
            @Override
            public String toString() {
                return filters.toQuery();
            }
        });
    }

    @NotNull
    private String getTable(SearchUrlsTable table, WebmasterHostId hostId) {
        return table.getFullTableName(getClickhouseServer(), hostId);
    }

    public static class F {
        public static final String HOST_ID = "host_id";
        public static final String PATH = "path";
        public static final String COPYRIGHT_OWNER = "copyright_owner";
        public static final String COPYRIGHT_OBJECT = "copyright_object";
    }
}
