package ru.yandex.webmaster3.storage.searchurl.samples.dao;

import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import lombok.RequiredArgsConstructor;
import lombok.Value;
import lombok.With;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.storage.util.clickhouse2.AbstractClickhouseDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHRow;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseQueryContext;
import ru.yandex.webmaster3.storage.util.clickhouse2.MdbClickhouseServer;
import ru.yandex.webmaster3.storage.util.clickhouse2.SimpleByteArrayOutputStream;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.Format;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.QueryBuilder;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.Statement;

/**
 * ishalaru
 * 17.03.2021
 **/
@Repository
@Slf4j
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class RealtimeUrlHttpCodesExamplesCHDao extends AbstractClickhouseDao {
    private static final int LIMIT_SAMPLES_PER_HOST = 10;
    public static final String DATABASE = DB_WEBMASTER3_SEARCHURLS;
    public static final String TABLE_NAME_POSTFIX = "realtime_url_http_codes_examples_distributed";
    private static final String TABLE_NAME = DATABASE + "." + TABLE_NAME_POSTFIX;
    @Qualifier("legacyMdbClickhouseServer")
    private final MdbClickhouseServer clickhouseServer;

    public void addRecord(List<Record> items) {

        if (items.size() == 0L) {
            return;
        }

        Statement st = QueryBuilder.insertInto(DATABASE, TABLE_NAME_POSTFIX)
                .fields(
                        F.DATE,
                        F.HOST_ID,
                        F.HOUR_INTERVAL,
                        F.URL,
                        F.HTTP_CODE,
                        F.LAST_ACCESS
                ).format(Format.Type.TAB_SEPARATED);
        String date = toClickhouseDate(LocalDate.now());

        try (SimpleByteArrayOutputStream bs = new SimpleByteArrayOutputStream()) {
            for (var item : items) {
                packRowValues(bs,
                        date,
                        item.getHostId(),
                        item.getHourInterval(),
                        item.getUrl(),
                        item.getHttpCode(),
                        item.getLastAccess()
                );
            }

            ClickhouseQueryContext.Builder context = ClickhouseQueryContext.useDefaults();
            clickhouseServer.insert(context, st.toString(), bs.toInputStream());
        } catch (IOException e) {
            throw new RuntimeException("Error when inserting realtime_url_http_codes records", e);
        }
    }

    public Map<WebmasterHostId, List<Record>> getSamples(long hourInterval, List<WebmasterHostId> hostIds,
                                                         long httpGroup) {
        if (hostIds.isEmpty()) {
            return Collections.emptyMap();
        }
        String query = QueryBuilder.selectDistinct(F.HOST_ID,
                F.HOUR_INTERVAL,
                F.URL,
                F.HTTP_CODE,
                F.LAST_ACCESS)
                .from(TABLE_NAME)
                .prewhere(QueryBuilder.eq(F.HOUR_INTERVAL, hourInterval))
                .and(QueryBuilder.in(F.HOST_ID, hostIds))
                .and(QueryBuilder.eq("intDiv(http_code, 100)", httpGroup))
                .limitBy(LIMIT_SAMPLES_PER_HOST, F.HOST_ID)
                .toString();

        return clickhouseServer.collectAll(query, Collectors.mapping(MAPPER, Collectors.groupingBy(Record::getHostId)));
    }

    private static final Function<CHRow, Record> MAPPER = chRow ->
            new Record(chRow.getHostId(F.HOST_ID), chRow.getLong(F.HOUR_INTERVAL), chRow.getString(F.URL),
                    chRow.getLong(F.HTTP_CODE), chRow.getLong(F.LAST_ACCESS));

    @Value
    @With
    public static class Record {
        WebmasterHostId hostId;
        long hourInterval;
        String url;
        long httpCode;
        long lastAccess;

        @JsonCreator
        public Record(@JsonProperty("hostId") WebmasterHostId hostId,
                      @JsonProperty("hourInterval") long hourInterval,
                      @JsonProperty("url") String url,
                      @JsonProperty("httpCode") long httpCode,
                      @JsonProperty("lastAccess") long lastAccess) {
            this.hostId = hostId;
            this.hourInterval = hourInterval;
            this.url = url;
            this.httpCode = httpCode;
            this.lastAccess = lastAccess;
        }

        public static Record of(String url, long hourInterval, long httpCode, long lastAccess) {
            return new Record(IdUtils.urlToHostId(url), hourInterval, url, httpCode, lastAccess);
        }
    }

    private interface F {
        String DATE = "date";
        String HOST_ID = "host_id";
        String HOUR_INTERVAL = "hour_interval";
        String URL = "url";
        String HTTP_CODE = "http_code";
        String LAST_ACCESS = "last_access";
    }


}
