package ru.yandex.webmaster3.storage.searchurl.samples.dao;

import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.metrika.counters.MetrikaCountersUtil;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.storage.AbstractFilter;
import ru.yandex.webmaster3.storage.TextFilterUtil;
import ru.yandex.webmaster3.storage.metrika.MetrikaCrawlStateService;
import ru.yandex.webmaster3.storage.searchurl.SearchUrlSampleField;
import ru.yandex.webmaster3.storage.searchurl.samples.dao.SearchUrlSamplesCHDao.F;
import ru.yandex.webmaster3.storage.searchurl.samples.data.SearchUrlCanonicalStatus;
import ru.yandex.webmaster3.storage.util.clickhouse2.condition.*;

import java.util.List;
import java.util.function.Predicate;

/**
 * @author avhaliullin
 */
@Component
public class SearchUrlConditions {
    private final MetrikaCrawlStateService metrikaCrawlStateService;

    @Autowired
    public SearchUrlConditions(MetrikaCrawlStateService metrikaCrawlStateService) {
        this.metrikaCrawlStateService = metrikaCrawlStateService;
    }

    public static Condition lastAccessCondition(Operator op, DateTime value) {
        return new TimestampCondition(F.LAST_ACCESS.getName(), op, value);
    }

    public static Condition titleContains(String value) {
        return new TextContainsCondition(F.TITLE.getName(), value);
    }

    public static Condition pathContains(String value) {
        return new TextContainsCondition(F.PATH.getName(), value);
    }

    public static TextLikeCondition buildPathCondition(TextLikeCondition.Builder builder) {
        return builder.build(SearchUrlEventSamplesCHDao.F.PATH.getName());
    }

    public static Condition canonicalStatusEquals(WebmasterHostId hostId, boolean fresh, SearchUrlCanonicalStatus canonicalStatus) {
        String hostUrl = IdUtils.hostIdToUrl(hostId);
        return new Condition(ConditionType.INT_EQUAL) {
            @Override
            public String toQuery() {
                switch (canonicalStatus) {
                    case UNDEFINED_CANONICAL:
                        return F.EX_REL_CANONICAL_TARGET.getName() + " == ''";
                    case CANONICAL:
                        return F.EX_REL_CANONICAL_TARGET.getName() + " == '" + hostUrl + "' || " + F.PATH.getName();
                    case NOT_CANONICAL:
                        return F.EX_REL_CANONICAL_TARGET.getName() + " <> '" + hostUrl + "' || " + F.PATH.getName() +
                                " AND " + F.EX_REL_CANONICAL_TARGET.getName() + " <> ''";
                    case FRESH_CANONICAL:
                        return fresh ? trueCondition().toQuery() : falseCondition().toQuery();
                }
                return null;
            }

            @Override
            public <T> Predicate<T> toPredicate(ConditionFieldExtractor<T> fieldExtractor) {
                return null;
            }
        };
    }

    public Condition makeCondition(WebmasterHostId hostId, boolean isFresh, List<? extends AbstractFilter<SearchUrlSampleField>> filters) {
        Condition condition = Condition.trueCondition();
        if (filters != null) {
            for (AbstractFilter<SearchUrlSampleField> filter : filters) {
                Operator dateOp = Operator.fromFilterOperation(filter.getOperation());

                switch (filter.getIndicator()) {
                    case TURBO:
                        condition = condition.andThen(new IntCondition(getTurboFieldName(), Operator.fromFilterOperation(filter.getOperation()), Integer.parseInt(filter.getValue())));
                        break;
                    case FROM_SITEMAP:
                        condition = condition.andThen(new IntCondition(getSitemapFieldName(), Operator.fromFilterOperation(filter.getOperation()), Integer.parseInt(filter.getValue())));
                        break;
                    case LAST_ACCESS_DATE:
                        if (dateOp == null) {
                            throw filter.invalidFilterException();
                        } else {
                            condition = condition.andThen(SearchUrlConditions.lastAccessCondition(dateOp, filter.parseDate()));
                        }
                        break;
                    case TITLE:
                        condition = condition.andThen(TextFilterUtil.getTextCondition(filter, SearchUrlConditions.getTitleFieldName()));
                        break;
                    case URL:
                        condition = condition.andThen(TextFilterUtil.getTextCondition(filter, SearchUrlConditions.getUrlFieldName()));
                        break;
                    case CANONICAL_STATUS:
                        if (isFresh) {
                            condition = condition.andThen(Condition.falseCondition());
                        } else {

                            if (filter.getOperation() == AbstractFilter.Operation.EQUAL) {
                                SearchUrlCanonicalStatus status = SearchUrlCanonicalStatus.valueOf(filter.getValue());
                                condition = condition.andThen(SearchUrlConditions.canonicalStatusEquals(hostId, isFresh, status));
                            } else {
                                throw filter.invalidFilterException();
                            }
                        }
                        break;
                    case VALID_FROM_METRIKA:
                        String domain = MetrikaCountersUtil.hostToPunycodeDomain(hostId);
                        boolean isMetrikaCrawlEnabled = metrikaCrawlStateService.getDomainCrawlStateCached(domain).hasEnabled();
                        condition = condition.andThen(ConditionUtils.makeValidFromMetrikaCondition(filter, isMetrikaCrawlEnabled, isFresh));
                        break;
                    default:
                        throw new RuntimeException("Not implemented filter for indicator " + filter.getIndicator());
                }
            }
        }
        return condition;
    }

    public static PathLikeConditionBuilder pathLikeConditionBuilder() {
        return new PathLikeConditionBuilder();
    }

    public static class PathLikeConditionBuilder {
        private final TextLikeCondition.Builder builder = TextLikeCondition.newBuilder();

        public PathLikeConditionBuilder append(String s) {
            builder.append(s);
            return this;
        }

        public PathLikeConditionBuilder appendWildcard() {
            builder.appendWildcard();
            return this;
        }

        public Condition build() {
            return builder.build(F.PATH.getName());
        }
    }

    public static String getTitleFieldName() {
        return F.TITLE.getName();
    }

    public static String getSitemapFieldName() {
        return F.EX_IS_FROM_SITEMAP.getName();
    }

    public static String getTurboFieldName() {
        return F.EX_IS_TURBO.getName();
    }

    public static String getUrlFieldName() {
        return F.PATH.getName();
    }
}
