package ru.yandex.webmaster3.storage.searchurl.samples.data;

import com.fasterxml.jackson.annotation.JsonProperty;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.util.TimeUtils;

/**
 * @author avhaliullin
 */
public class SearchUrlEventSample {
    private final String path;
    private final String url;
    private final String title;
    @Nullable
    private final LocalDate lastAccess;
    @NotNull
    private final LocalDateTime searchBaseDate;
    private final SearchUrlEventType eventType;
    private final UrlStatusInfo statusInfo;
    private final SearchUrlCanonicalStatus canonicalStatus;

    public SearchUrlEventSample(@JsonProperty("path") String path,
                                @JsonProperty("url") String url,
                                @JsonProperty("title") String title,
                                @Nullable @JsonProperty("lastAccess") LocalDate lastAccess,
                                @NotNull @JsonProperty("searchBaseDate") LocalDateTime searchBaseDate,
                                @JsonProperty("eventType") SearchUrlEventType eventType,
                                @JsonProperty("statusInfo") UrlStatusInfo statusInfo) {
        this.path = path;
        this.url = url;
        this.title = title;
        this.lastAccess = lastAccess;
        this.searchBaseDate = searchBaseDate;
        this.eventType = eventType;
        this.statusInfo = statusInfo;
        this.canonicalStatus = SearchUrlCanonicalStatus.fromUrlAndCanonical(url, statusInfo);
    }

    public SearchUrlEventSample(String path,
                                String url,
                                String title,
                                LocalDate lastAccess,
                                LocalDateTime searchBaseDate,
                                SearchUrlEventType eventType,
                                UrlStatusInfo statusInfo,
                                SearchUrlCanonicalStatus canonicalStatus) {
        this.path = path;
        this.url = url;
        this.title = title;
        this.lastAccess = lastAccess;
        this.searchBaseDate = searchBaseDate;
        this.eventType = eventType;
        this.statusInfo = statusInfo;
        this.canonicalStatus = canonicalStatus;
    }

    @Description("относительный адрес страницы (без хоста)")
    public String getPath() {
        return path;
    }

    @Description("адрес страницы")
    public String getUrl() {
        return url;
    }

    @Description("заголовок страницы")
    public String getTitle() {
        return title;
    }

    @Nullable
    @Description("дата последнего посещения страницы")
    public LocalDate getLastAccess() {
        return lastAccess;
    }

    @NotNull
    @Description("дата базы, в которой произошло событие")
    public LocalDate getSearchBaseDate() {
        return searchBaseDate.toLocalDate();
    }

    @NotNull
    @Description("дата и время базы, в которой произошло событие")
    public DateTime getSearchBaseDateTime() {
        return searchBaseDate.toDateTime(TimeUtils.EUROPE_MOSCOW_ZONE);
    }

    @Description("что случилось со страницей")
    public SearchUrlEventType getEventType() {
        return eventType;
    }

    @Description("информация о статусе страницы на поиске")
    public UrlStatusInfo getStatusInfo() {
        return statusInfo;
    }

    public SearchUrlCanonicalStatus getCanonicalStatus() {
        return canonicalStatus;
    }
}
