package ru.yandex.webmaster3.storage.searchurl.samples.data;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import org.jetbrains.annotations.Nullable;
import org.joda.time.DateTime;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.sitestructure.SearchUrlStatusEnum;

/**
 * @author avhaliullin
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public class UrlStatusInfo {
    private final SearchUrlStatusEnum status;
    private final DateTime addTime;
    private final String beautyUrl;
    private final Integer httpCode;
    private final String mainHost;
    private final String mainPath;
    private final String redirectTarget;
    private final String relCanonicalTarget;
    private final String description;
    private final Boolean isTurboPage;
    private final Boolean isFromSitemap;
    private final Boolean isValidFromMetrika;
    private final Boolean isValidFromIndexNow;
    private final Boolean isFresh;

    @JsonCreator
    public UrlStatusInfo(
            @JsonProperty("status") SearchUrlStatusEnum status,
            @JsonProperty("addTime") DateTime addTime,
            @JsonProperty("beautyUrl") String beautyUrl,
            @JsonProperty("httpCode") Integer httpCode,
            @JsonProperty("mainHost") String mainHost,
            @JsonProperty("mainPath") String mainPath,
            @JsonProperty("redirectTarget") String redirectTarget,
            @JsonProperty("relCanonicalTarget") String relCanonicalTarget,
            @JsonProperty("description") String description,
            @JsonProperty("isTurboPage") Boolean isTurboPage,
            @JsonProperty("isFromSitemap") Boolean isFromSitemap,
            @JsonProperty(value = "isValidFromMetrika", defaultValue = "false") Boolean isValidFromMetrika,
            @JsonProperty(value = "isValidFromIndexNow", defaultValue = "false") Boolean isValidFromIndexNow,
            @JsonProperty(value = "fresh", defaultValue = "false") Boolean isFresh) {
        this.status = status;
        this.addTime = addTime;
        this.beautyUrl = beautyUrl;
        this.httpCode = httpCode;
        this.mainHost = mainHost;
        this.mainPath = mainPath;
        this.redirectTarget = redirectTarget;
        this.relCanonicalTarget = relCanonicalTarget;
        this.description = description;
        this.isTurboPage = isTurboPage;
        this.isFromSitemap = isFromSitemap;
        this.isValidFromIndexNow = isValidFromIndexNow;
        this.isFresh = isFresh;
        this.isValidFromMetrika = isValidFromMetrika;
    }


    @Description("статус страницы в поиске")
    public SearchUrlStatusEnum getStatus() {
        return status;
    }

    public DateTime getAddTime() {
        return addTime;
    }

    public String getBeautyUrl() {
        return beautyUrl;
    }

    public Integer getHttpCode() {
        return httpCode;
    }

    public String getMainHost() {
        return mainHost;
    }

    public String getMainPath() {
        return mainPath;
    }

    public String getRedirectTarget() {
        return redirectTarget;
    }

    public String getRelCanonicalTarget() {
        return relCanonicalTarget;
    }

    public String getDescription() {
        return description;
    }

    public Boolean isTurboPage() {
        return isTurboPage;
    }

    public Boolean isFromSitemap() {
        return isFromSitemap;
    }

    public Boolean isFresh() {
        return isFresh;
    }

    public Boolean isValidFromMetrika() {
        return isValidFromMetrika;
    }

    public Boolean getValidFromIndexNow() {
        return isValidFromIndexNow;
    }

    @Nullable
    @JsonIgnore
    public String getTargetUrl() {
        switch (status) {
            case INDEXED_NOTSEARCHABLE:
            case SEMIDUP:
                return beautyUrl;

            case REDIRECT_NOTSEARCHABLE:
                return redirectTarget;

            case NOT_CANONICAL:
                return relCanonicalTarget;

            default:
                return null;
        }
    }

    public static boolean isValidFromMetrika(long validFromMetrikaLastAccess, long validFromIndexNowLastAccess) {
        if (validFromMetrikaLastAccess == 0) {
            return false;
        } else if (validFromIndexNowLastAccess == 0) {
            return true;
        } else {
            return validFromMetrikaLastAccess > validFromIndexNowLastAccess;
        }
    }

    public static boolean isValidFromIndexNow(long validFromMetrikaLastAccess, long validFromIndexNowLastAccess) {
        if (validFromIndexNowLastAccess == 0) {
            return false;
        } else if (validFromMetrikaLastAccess == 0) {
            return true;
        } else {
            return validFromIndexNowLastAccess > validFromMetrikaLastAccess;
        }
    }
}
