package ru.yandex.webmaster3.storage.serplinks;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.http.HttpStatus;
import org.apache.http.NameValuePair;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.HttpConstants;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.metrics.externals.AbstractExternalAPIService;
import ru.yandex.webmaster3.core.metrics.externals.ExternalDependencyMethod;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.JavaMethodWitness;
import ru.yandex.webmaster3.storage.serplinks.data.SerpLinkUpdateInfo;
import ru.yandex.webmaster3.storage.serplinks.data.SerpLinksSectionId;
import ru.yandex.wmtools.common.Constants;
import ru.yandex.wmtools.common.SupportedProtocols;
import ru.yandex.wmtools.common.service.IService;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author Andrey Mima (amima@yandex-team.ru)
 */
public class W3SerpLinksService extends AbstractExternalAPIService implements IService, Constants {
    private static final Logger log = LoggerFactory.getLogger(W3SerpLinksService.class);
    private static final ObjectMapper OM = new ObjectMapper()
            .setSerializationInclusion(JsonInclude.Include.NON_NULL);

    private static final String PARAM_ACTION = "action";

    private static final String TAG_HOST = "host";
    private static final String TAG_PAGE = "page";
    private static final String TAG_NAME = "name";
    private static final String ATTR_URL = "url";
    private static final String ATTR_SORTBY = "sortby";
    private static final String ATTR_BLOCK = "block";
    private static final String ATTR_WEIGHT = "weight";
    private static final String ATTR_HIDING_REASON = "hiding-reason";
    private static final String ATTR_CODE = "code";

    private static final String CODE_NO_SUCH_PAGE = "NO_SUCH_PAGE";
    private static final String CODE_NOT_ALLOWED_NAME = "NOT_ALLOWED_NAME";
    private static final String CODE_NO_SITELINKS = "NO_SITELINKS";
    private static final String CODE_CHILDREN_ERROR = "CHILDREN_ERROR";

    private static final int SOCKET_TIMEOUT = (int) TimeUnit.SECONDS.toMillis(10);

    private String host;
    private String path;

    public SerpLinkInfo getSerpLinkInfoDiff(SerpLinksSectionId sectionId) {
        return getSerpLinkInfo(sectionId, Action.DIFF);
    }

    public SerpLinkInfo getSerpLinkInfo(SerpLinksSectionId sectionId) {
        return getSerpLinkInfo(sectionId, Action.INFO);
    }

    private UpdateResult checkResponse(Element root) {
        String errorCode = root.getAttribute(ATTR_CODE);
        switch (errorCode) {
            case CODE_NO_SUCH_PAGE:
            case CODE_NOT_ALLOWED_NAME:
                return UpdateResult.EXPIRED;
            case CODE_NO_SITELINKS:
                return UpdateResult.SECTION_DOESNT_EXIST;
            case CODE_CHILDREN_ERROR:
                NodeList nodeList = root.getChildNodes();
                for (int i = 0; i < nodeList.getLength(); i++) {
                    Node node = nodeList.item(i);
                    if (node instanceof Element) {
                        Element element = (Element) node;
                        if (element.hasAttribute(ATTR_CODE)) {
                            UpdateResult childResult = checkResponse(element);
                            if (childResult != UpdateResult.OK) {
                                return childResult;
                            }
                        }
                    }
                }
                throw new WebmasterException("Serplinks service returned CHILDREN_ERROR, but no children with error were found ",
                        new WebmasterErrorResponse.SerpLinksErrorResponse(this.getClass(), null), null);
            default:
                throw new WebmasterException("Unknown serplinks service error: " + errorCode,
                        new WebmasterErrorResponse.SerpLinksErrorResponse(this.getClass(), null), null);
        }
    }

    @ExternalDependencyMethod("update")
    public UpdateResult updateInfo(SerpLinksSectionId sectionId, Sort sort, List<SerpLinkUpdateInfo> links, boolean block) {
        return trackQuery(new JavaMethodWitness() {}, ALL_ERRORS_INTERNAL, () -> {
            try {
                Map<String, String> postSitelinks = new HashMap<>();
                List<SerializableLinkInfo> toSerialization = new ArrayList<>();
                if (links != null && !links.isEmpty()) {
                    for (SerpLinkUpdateInfo info : links) {
                        toSerialization.add(new SerializableLinkInfo(info.getId(), info.getName(), info.isBlocked()));
                    }
                    postSitelinks.put("changes", OM.writer().writeValueAsString(toSerialization));
                }
                Document result = serpLinksRequest(Action.EDIT_SITELINKS, sectionId, postSitelinks, Param.noEncode,
                        Param.stringParam(Edit.SORT, sort == null ? null : sort.getValue()),
                        Param.booleanParam(Edit.BLOCK, block));
                String status = result.getDocumentElement().getAttribute("status");
                if ("ok".equals(status)) {
                    return UpdateResult.OK;
                }

                return checkResponse(result.getDocumentElement());
            } catch (JsonProcessingException e) {
                throw new WebmasterException("Failed to serialize serplinks edit request",
                        new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), "Failed to serialize serplinks edit request"), e);
            }
        });
    }

    @ExternalDependencyMethod(("sections"))
    public List<String> getSections(WebmasterHostId hostId) {
        return trackQuery(new JavaMethodWitness() {}, ALL_ERRORS_INTERNAL, () -> {
            Document response = serpLinksRequest(Action.SECTIONS, new SerpLinksSectionId(hostId, ""), null, Param.noEncode);
            NodeList list = response.getDocumentElement().getChildNodes();
            List<String> results = new LinkedList<>();
            for (int i = 0; i < list.getLength(); i++) {
                Node host = list.item(i);
                if (TAG_HOST.equals(host.getNodeName())) {
                    String url = host.getAttributes().getNamedItem(ATTR_URL).getNodeValue();
                    //В теге host всегда приходит урл без протокола, поэтому относительный урл получаем легко и надежно
                    int slashIndex = url.indexOf('/');
                    String relUrl;
                    if (slashIndex < 0) {
                        relUrl = "/";
                    } else {
                        relUrl = url.substring(slashIndex);
                    }
                    results.add(relUrl);
                }
            }
            return results;
        });
    }

    private String getSectionName(SerpLinksSectionId hostId) {
        return stripTrailingSlash(IdUtils.toRobotHostString(hostId.getHostId()) + hostId.getSection());
    }

    @ExternalDependencyMethod("get-info")
    private SerpLinkInfo getSerpLinkInfo(SerpLinksSectionId hostName, Action action) {
        return trackQuery(new JavaMethodWitness() {}, ALL_ERRORS_INTERNAL, () -> {
            Document response = serpLinksRequest(action, hostName, null, Param.noEncode);
            Node sortNode = response.getDocumentElement().getFirstChild().getAttributes().getNamedItem(ATTR_SORTBY);
            boolean isAlphaSort = sortNode != null && "alpha".equals(sortNode.getTextContent());
            Node blockedNode = response.getDocumentElement().getFirstChild().getAttributes().getNamedItem(ATTR_BLOCK);
            boolean blocked = blockedNode != null && "true".equals(blockedNode.getTextContent());
            NodeList upLevel = response.getDocumentElement().getFirstChild().getChildNodes();
            List<SerpLinkInfo.SerpLinksPage> pages = new ArrayList<>();
            for (int i = 0; i < upLevel.getLength(); i++) {
                Node pageLevelNode = upLevel.item(i);
                if (TAG_PAGE.equals(pageLevelNode.getNodeName())) {
                    SerpLinkInfo.HideStateEnum hideState = SerpLinkInfo.HideStateEnum.NOT_HIDDEN;
                    Node hidingReasonAttr = pageLevelNode.getAttributes().getNamedItem(ATTR_HIDING_REASON);
                    if (hidingReasonAttr != null) {
                        hideState = SerpLinkInfo.HideStateEnum.getByValue(hidingReasonAttr.getNodeValue());
                    }
                    SerpLinkInfo.ModificationStatusEnum pageModificationStatus = getModificationStatus(pageLevelNode);
                    String pageId = pageLevelNode.getAttributes().getNamedItem(ATTR_URL).getNodeValue();
                    URL pageUrl;
                    try {
                        pageUrl = SupportedProtocols.getURL(pageId);
                    } catch (MalformedURLException | URISyntaxException | SupportedProtocols.UnsupportedProtocolException e) {
                        throw new WebmasterException("Unable to process URL from serplinks response " + pageId,
                                new WebmasterErrorResponse.SerpLinksErrorResponse(this.getClass(), e), e);
                    }
                    SerpLinkInfo.SerpLinksPage serpLink = new SerpLinkInfo.SerpLinksPage(
                            pageId, pageUrl,
                            pageLevelNode.getAttributes().getNamedItem(ATTR_BLOCK) != null,
                            Float.valueOf(pageLevelNode.getAttributes().getNamedItem(ATTR_WEIGHT).getNodeValue()),
                            hideState, pageModificationStatus);
                    NodeList secondLevel = pageLevelNode.getChildNodes();
                    for (int j = 0; j < secondLevel.getLength(); j++) {
                        Node nameLevelNode = secondLevel.item(j);
                        if (TAG_NAME.equals(nameLevelNode.getNodeName())) {
                            String weight = nameLevelNode.getAttributes().getNamedItem(ATTR_WEIGHT).getNodeValue();
                            String name = nameLevelNode.getTextContent();
                            SerpLinkInfo.ModificationStatusEnum nameModificationStatus = getModificationStatus(nameLevelNode);
                            serpLink.addPageName(new SerpLinkInfo.SerpLinksPageName(name, Float.valueOf(weight), nameModificationStatus));
                        }
                    }
                    pages.add(serpLink);
                }
            }

            return new SerpLinkInfo(blocked, isAlphaSort, pages);
        });
    }

    private SerpLinkInfo.ModificationStatusEnum getModificationStatus(Node node) {
        Node deletedAttr = node.getAttributes().getNamedItem("deleted");
        if (deletedAttr != null && "true".equals(deletedAttr.getNodeValue())) {
            return SerpLinkInfo.ModificationStatusEnum.DELETED;
        }

        Node newAttr = node.getAttributes().getNamedItem("new");
        if (newAttr != null && "true".equals(newAttr.getNodeValue())) {
            return SerpLinkInfo.ModificationStatusEnum.NEW;
        }

        return SerpLinkInfo.ModificationStatusEnum.NOT_CHANGED;
    }

    private Document serpLinksRequest(Action action, SerpLinksSectionId hostId, Map<String, String> postParams, Param... params) {
        HttpUriRequest request;
        try {
            URIBuilder uriBuilder = new URIBuilder()
                    .setScheme("http")
                    .setHost(host)
                    .setPath(path)
                    .addParameter(PARAM_ACTION, action.getValue())
                    .addParameter(Params.HOST.getValue(), getSectionName(hostId));
            for (Param param : params) {
                uriBuilder.addParameter(param.name.getValue(), param.value);
            }

            if (postParams == null) {
                request = new HttpGet(uriBuilder.build());
                log.debug("Requesting serplinks with GET on {}", uriBuilder);
            } else {
                HttpPost pm = new HttpPost(uriBuilder.build());
                if (log.isDebugEnabled()) {
                    StringBuilder message = new StringBuilder(
                            "Requesting serplinks with POST on " + uriBuilder + " and post params: ");
                    for (Map.Entry<String, String> param : postParams.entrySet()) {
                        message.append(param.getKey()).append(" = ").append(param.getValue()).append("; ");
                    }
                    log.debug(message.toString());
                }

                List<NameValuePair> nameValuePairs = postParams.entrySet().stream()
                        .map(param -> new BasicNameValuePair(param.getKey(), param.getValue()))
                        .collect(Collectors.toList());
                UrlEncodedFormEntity entity = new UrlEncodedFormEntity(nameValuePairs, StandardCharsets.UTF_8);
                pm.setEntity(entity);
                request = pm;
            }
        } catch (URISyntaxException e) {
            throw new WebmasterException("SerpLinks request failed",
                    new WebmasterErrorResponse.SerpLinksErrorResponse(this.getClass(), e), e);
        }
        try {
            RequestConfig requestConfig = RequestConfig.custom()
                    .setConnectTimeout(HttpConstants.DEFAULT_CONNECT_TIMEOUT)
                    .setSocketTimeout(SOCKET_TIMEOUT)
                    .build();

            try (CloseableHttpClient httpClient = HttpClientBuilder.create()
                    .setDefaultRequestConfig(requestConfig).build();
                 CloseableHttpResponse httpResponse = httpClient.execute(request)) {
                if (httpResponse.getStatusLine().getStatusCode() != HttpStatus.SC_OK) {
                    throw new WebmasterException("Serplinks returned http status " + httpResponse.getStatusLine(),
                            new WebmasterErrorResponse.SerpLinksErrorResponse(this.getClass(), null));
                }
                String responseStr = EntityUtils.toString(httpResponse.getEntity(), StandardCharsets.UTF_8);
                log.debug("Serplinks response:\n{}", responseStr);

                DocumentBuilder documentBuilder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
                Document result = documentBuilder.parse(new ByteArrayInputStream(responseStr.getBytes(StandardCharsets.UTF_8)));
                return result;
            }
        } catch (IOException e) {
            throw new WebmasterException("SerpLinks request failed",
                    new WebmasterErrorResponse.SerpLinksErrorResponse(this.getClass(), e), e);
        } catch (ParserConfigurationException | SAXException e) {
            throw new WebmasterException("SerpLinks response processing failed",
                    new WebmasterErrorResponse.SerpLinksErrorResponse(this.getClass(), e), e);
        }
    }

    private String stripTrailingSlash(String s) {
        if (s.endsWith("/")) {
            return s.substring(0, s.length() - 1);
        } else {
            return s;
        }
    }

    @Required
    public void setHost(String host) {
        this.host = host;
    }

    @Required
    public void setPath(String path) {
        this.path = path;
    }

    interface ParamName {
        String getValue();
    }

    private static class Param {
        public final ParamName name;
        public final String value;

        private Param(ParamName name, String value) {
            this.name = name;
            this.value = value;
        }

        StringBuilder append(StringBuilder sb) {
            if (value != null) {
                return sb.append("&").append(name.getValue()).append("=").append(value);
            } else {
                return sb;
            }
        }

        public static Param stringParam(ParamName name, String value) {
            return new Param(name, value);
        }

        public static Param booleanParam(ParamName name, boolean value) {
            return new Param(name, String.valueOf(value));
        }

        public static final Param checkStrict = stringParam(Params.CHECK, "strict");
        public static final Param noEncode = booleanParam(Params.ENCODE, false);
    }

    private enum Action {
        VIEW("view"),
        EDIT("edit"),
        EDIT_SITELINKS("editSitelinks"),
        SECTIONS("sections"),
        INFO("info"),
        DIFF("diff");

        private final String value;

        private Action(String value) {
            this.value = value;
        }

        public String getValue() {
            return value;
        }
    }

    private enum Params implements ParamName {
        HOST("host"),
        PAGE("page"),
        NAME("name"),
        CHECK("check"),
        ENCODE("encode");

        private final String value;

        private Params(String value) {
            this.value = value;
        }

        public String getValue() {
            return value;
        }
    }

    private enum Edit implements ParamName {
        BLOCK("blockHost"),
        REMOVE("remove"),
        SORT("sortby");

        private final String value;

        private Edit(String value) {
            this.value = value;
        }

        public String getValue() {
            return value;
        }
    }

    public enum Sort {
        ALPHA("alpha"),
        WEIGHT("weight");

        private final String value;

        private Sort(String value) {
            this.value = value;
        }

        public String getValue() {
            return value;
        }

        public boolean isAlphaSort() {
            return this == ALPHA;
        }
    }

    private class SerializableLinkInfo {
        private final String url;
        private final String name;
        private final Boolean remove;

        public SerializableLinkInfo(String url, String name, Boolean remove) {
            this.url = url;
            this.name = name;
            this.remove = remove;
        }

        public String getUrl() {
            return url;
        }

        public String getName() {
            return name;
        }

        public Boolean getRemove() {
            return remove;
        }
    }

    public enum UpdateResult {
        OK,
        EXPIRED,
        SECTION_DOESNT_EXIST
    }
}
