package ru.yandex.webmaster3.storage.services;

import java.io.IOException;
import java.util.Collection;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import lombok.Setter;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.util.W3Collectors;
import ru.yandex.webmaster3.storage.clickhouse.ClickhouseTableInfo;
import ru.yandex.webmaster3.storage.clickhouse.TableProvider;
import ru.yandex.webmaster3.storage.clickhouse.TableType;
import ru.yandex.webmaster3.storage.util.JsonDBMapping;
import ru.yandex.webmaster3.storage.util.clickhouse2.AbstractClickhouseDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHRow;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.QueryBuilder;

public class SiteServicesCHDao extends AbstractClickhouseDao {

    @Setter
    private TableProvider tableStorage;

    private static final String[] COLUMNS = {
            F.OWNER, F.ALICE_INFO, F.COMPANY_INFO, F.MOBILE_APP_INFO, F.IS_SHOP,
            F.MARKET_RATING, F.MARKET_ID, F.USE_DIRECT, F.USE_METRIKA, F.RADAR_TYPE, F.SNIPPET_PIC, F.IS_NEWS,
            F.VERTICAL_SHARE_IS_VALIDATED, F.VERTICAL_SHARE_RATING
    };

    private static final String[] COLUMNS_RIVALS = {
            F.OWNER, F.COMPANY_INFO, F.MARKET_RATING, F.MARKET_ID, F.USE_DIRECT, F.USE_METRIKA, F.MOBILE_APP_INFO,
            F.VERTICAL_SHARE_IS_VALIDATED, F.VERTICAL_SHARE_RATING
    };


    public Map<SiteServiceType, ServiceInfo> getRivalServiceInfo(String owner) {
        return getSiteServices(Set.of(owner), COLUMNS_RIVALS).get(owner);
    }

    public Map<String, Map<SiteServiceType, ServiceInfo>> getRivalServiceInfo(Collection<String> owners) {
        return getSiteServices(owners, COLUMNS_RIVALS);
    }

    public Map<SiteServiceType, ServiceInfo> getSiteServiceInfo(String owner) {
        return getSiteServices(Set.of(owner), COLUMNS).get(owner);
    }

    public Map<String, Map<SiteServiceType, ServiceInfo>> getSiteServiceInfo(Collection<String> owners) {
        return getSiteServices(owners, COLUMNS);
    }

    public Map<String, Map<SiteServiceType, ServiceInfo>> getSiteServices(Collection<String> owners, String[] columns) {
        if (owners.isEmpty()) {
            return new HashMap<>();
        }
        ClickhouseTableInfo table = tableStorage.getTable(TableType.SITE_SERVICES);

        String query = QueryBuilder.select(columns)
                .from(table.getLocalTableName())
                .where(QueryBuilder.in(F.OWNER, owners))
                .toString();

        Mapper mapper = new Mapper(columns);

        Map<String, Map<SiteServiceType, ServiceInfo>> siteservicesByOwner = getClickhouseServer()
                .collectAll(query, Collectors.mapping(mapper::applyWithOwner,
                        W3Collectors.toHashMap(W3Collectors.replacingMerger())));

        owners.forEach(owner ->
                siteservicesByOwner.putIfAbsent(owner, new EnumMap<>(SiteServiceType.class)));
        return siteservicesByOwner;
    }

    public interface F {
        String OWNER = "owner";
        String ALICE_INFO = "alice_info";
        String COMPANY_INFO = "company_info";
        String MOBILE_APP_INFO = "mobile_app_info";
        String IS_SHOP = "is_shop";
        String MARKET_RATING = "market_rating";
        String MARKET_ID = "market_id";
        String USE_DIRECT = "use_direct";
        String USE_METRIKA = "use_metrika";
        String RADAR_TYPE = "radar_type";
        String SNIPPET_PIC = "snippet_pic";
        String IS_NEWS = "is_news";
        String VERTICAL_SHARE_RATING = "vertical_share_rating";
        String VERTICAL_SHARE_IS_VALIDATED = "vertical_share_is_validated";
    }

    private static final class Mapper implements Function<CHRow, Map<SiteServiceType, ServiceInfo>> {
        private final Set<String> columns;

        public Mapper(String[] columns) {
            this.columns = Set.of(columns);
        }

        @Override
        public Map<SiteServiceType, ServiceInfo> apply(CHRow chRow) {
            try {
                return applyInternal(chRow);
            } catch (IOException e) {
                throw new WebmasterException("Wrong json",
                        new WebmasterErrorResponse.UnableToReadJsonRequestResponse(getClass(), e), e);
            }
        }

        public Pair<String, Map<SiteServiceType, ServiceInfo>> applyWithOwner(CHRow chRow) {
            try {
                return Pair.of(getNullableString(chRow, F.OWNER), applyInternal(chRow));
            } catch (IOException e) {
                throw new WebmasterException("Wrong json",
                        new WebmasterErrorResponse.UnableToReadJsonRequestResponse(getClass(), e), e);
            }
        }

        private Map<SiteServiceType, ServiceInfo> applyInternal(CHRow chRow) throws IOException {
            Map<SiteServiceType, ServiceInfo> services = new EnumMap<>(SiteServiceType.class);

            if (columns.contains(F.ALICE_INFO)) {
                String aliceInfoJson = chRow.getString(F.ALICE_INFO);
                if (!Strings.isNullOrEmpty(aliceInfoJson)) {
                    services.put(SiteServiceType.ALICE_DIALOGS,
                            new AliceDialogsInfo(
                                    JsonDBMapping.OM.readValue(
                                            aliceInfoJson,
                                            AliceDialogsInfo.TYPE_FACTORY_LIST
                                    )
                            ));
                }
            }

            if (columns.contains(F.MOBILE_APP_INFO)) {
                String mobileAppJson = chRow.getString(F.MOBILE_APP_INFO);
                if (!Strings.isNullOrEmpty(mobileAppJson)) {
                    services.put(
                            SiteServiceType.MOBILE_APP,
                            new MobileAppInfo(
                                    JsonDBMapping.OM.readValue(
                                            mobileAppJson,
                                            MobileAppInfo.TYPE_FACTORY_MAP
                                    )
                            ));
                }
            }

            if (columns.contains(F.RADAR_TYPE)) {
                String radarType = chRow.getString(F.RADAR_TYPE);
                if (!Strings.isNullOrEmpty(radarType)) {
                    services.put(SiteServiceType.RADAR_TYPE, new RadarTypeInfo(radarType));
                }
            }

            if (columns.contains(F.MARKET_ID) && columns.contains(F.MARKET_RATING)) {
                long marketId = chRow.getLongUnsafe(F.MARKET_ID);
                double marketRating = chRow.getDouble(F.MARKET_RATING);
                if (marketId != 0 && marketRating != 0) {
                    services.put(SiteServiceType.MARKET_RATING, new MarketRatingInfo(marketId, marketRating));
                }
            }

            if (columns.contains(F.COMPANY_INFO)) {
                String companyInfo = chRow.getString(F.COMPANY_INFO);
                if (!Strings.isNullOrEmpty(companyInfo)) {
                    CompanyInfo cmp = JsonDBMapping.OM.readValue(companyInfo, CompanyInfo.class);
                    services.put(SiteServiceType.COMPANY_INFO, cmp);
                }
            }
            if (columns.contains(F.IS_SHOP)) {
                services.put(SiteServiceType.IS_SHOP_HOST, new IsShopHostInfo(chRow.getInt(F.IS_SHOP) == 1));
            }
            if (columns.contains(F.USE_DIRECT)) {
                services.put(SiteServiceType.USE_DIRECT, new UseDirectInfo(chRow.getInt(F.USE_DIRECT) == 1));
            }
            if (columns.contains(F.USE_METRIKA)) {
                services.put(SiteServiceType.USE_METRIKA, new UseMetrikaInfo(chRow.getInt(F.USE_METRIKA) == 1));
            }

            if (columns.contains(F.SNIPPET_PIC)) {
                services.put(SiteServiceType.SNIPPET_PIC, new HaveSnippetPicture(chRow.getInt(F.SNIPPET_PIC) == 1));
            }
            if (columns.contains(F.IS_NEWS)) {
                services.put(SiteServiceType.IS_NEWS, new IsNews(chRow.getInt(F.IS_NEWS) == 1));
            }
            if (columns.contains(F.VERTICAL_SHARE_RATING)) {
                double rating = chRow.getDouble(F.VERTICAL_SHARE_RATING);
                if (rating != 0) {
                    services.put(SiteServiceType.VERTICAL_SHARE_RATING, new VerticalShareRatingInfo(rating));
                }
            }
            if (columns.contains(F.VERTICAL_SHARE_IS_VALIDATED)) {
                boolean isValidated = chRow.getInt(F.VERTICAL_SHARE_IS_VALIDATED) > 0;
                services.put(SiteServiceType.VERTICAL_SHARE_IS_VALIDATED, new VerticalShareIsValidatedInfo(isValidated));
            }
            return services;
        }
    }
}
