package ru.yandex.webmaster3.storage.settings.history;

import java.util.List;

import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Value;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.turbo.model.TurboHostSettings;
import ru.yandex.webmaster3.core.turbo.model.desktop.TurboDesktopSettings;
import ru.yandex.webmaster3.storage.user.notification.Setting;
import ru.yandex.webmaster3.storage.util.JsonDBMapping;

@Service("settingsHistoryService")
@AllArgsConstructor(onConstructor_ = @Autowired)
public final class SettingsHistoryService {

    private final UserSettingsHistoryYDao userSettingsHistoryYDao;
    private final TurboSettingsHistoryYDao turboSettingsHistoryYDao;

    public void updateNotificationEmail(Long userId, String email) {
        LogSettingInfo info = LogSettingInfo.builder()
                .userId(userId)
                .type(SettingsHistoryType.EMAIL)
                .newState(email)
                .updateDate(DateTime.now())
                .build();
        userSettingsHistoryYDao.insert(info);
    }

    public void updateHostNotificationSetting(Long userId, WebmasterHostId webmasterHostId, Setting setting) {
        LogSettingInfo info = LogSettingInfo.builder()
                .userId(userId)
                .type(SettingsHistoryType.HOST_NOTIFICATIONS)
                .hostId(webmasterHostId.toStringId())
                .newState(JsonDBMapping.writeValueAsString(setting))
                .updateDate(DateTime.now())
                .build();
        userSettingsHistoryYDao.insert(info);
    }

    public void updateHostNotificationDefault(Long userId, Setting setting) {
        LogSettingInfo info = LogSettingInfo.builder()
                .userId(userId)
                .type(SettingsHistoryType.USER_NOTIFICATIONS)
                .newState(JsonDBMapping.writeValueAsString(setting))
                .updateDate(DateTime.now())
                .build();
        userSettingsHistoryYDao.insert(info);
    }

    public void updateSettingsTurbo(Long userId, String domain, TurboHostSettings turboSettings) {
        LogSettingInfo info = LogSettingInfo.builder()
                .userId(userId)
                .hostId(domain)
                .type(SettingsHistoryType.TURBO_DOMAIN)
                .newState(JsonDBMapping.writeValueAsString(turboSettings))
                .updateDate(DateTime.now())
                .build();
        turboSettingsHistoryYDao.insert(info);
    }

    public void updateSettingsTurbo(Long userId, String domain, TurboDesktopSettings turboSettings) {
        LogSettingInfo info = LogSettingInfo.builder()
                .userId(userId)
                .hostId(domain)
                .type(SettingsHistoryType.TURBO_DESKTOP)
                .updateDate(DateTime.now())
                .newState(JsonDBMapping.writeValueAsString(turboSettings)).build();
        turboSettingsHistoryYDao.insert(info);
    }

    public List<LogSettingInfo> turboChangesForHostAndType(String hostId, SettingsHistoryType type) {
        return turboSettingsHistoryYDao.changesForHostAndType(hostId, type);
    }

    @Value
    @Builder
    public static class LogSettingInfo {
        Long userId;
        SettingsHistoryType type;
        String newState;
        @Builder.Default
        String hostId = "";
        DateTime updateDate;
    }
}
