package ru.yandex.webmaster3.storage.sitemap;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.util.sitemap.SitemapRecrawlQuotaUtil;
import ru.yandex.webmaster3.core.worker.client.WorkerClient;
import ru.yandex.webmaster3.core.worker.task.SitemapRecrawlTaskData;
import ru.yandex.webmaster3.storage.sitemap.dao.SitemapRecrawlRequest;
import ru.yandex.webmaster3.storage.sitemap.dao.SitemapRecrawlRequestsYDao;

/**
 * Created by leonidrom on 23/05/2017.
 */
@Slf4j
@Service
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class SitemapRecrawlRequestService {

    public static final int DAYS_IN_MONTH = 30;
    public static final int RECRAWL_SITEMAP_QUOTA_PER_MONTH = 10;

    private final WorkerClient workerClient;
    private final SitemapRecrawlRequestsYDao sitemapRecrawlRequestsYDao;

    public DateTime createRequest(WebmasterHostId hostId, UUID sitemapId, UUID parentSitemapId, String sitemapUrl) {
        DateTime requestDate = DateTime.now();
        sitemapRecrawlRequestsYDao.storeRequest(hostId, sitemapId, parentSitemapId, sitemapUrl, requestDate);
        workerClient.enqueueTask(new SitemapRecrawlTaskData(sitemapUrl));
        return requestDate;
    }

    /**
     * Возвращает пару: мапу из пар (sitemapId, дата последнего запроса на переобход) и флаг - влезет ли еще один запрос в хостовую квоту
     */
    public Pair<Map<UUID, DateTime>, Boolean> getLastRequestsInfo(WebmasterHostId hostId) {
        List<Pair<UUID, DateTime>> recrawlRequests = sitemapRecrawlRequestsYDao.getRequestTimes(hostId);
        Map<UUID, DateTime> uuidToLastRecrawlRequest = new HashMap<>(recrawlRequests.size());
        recrawlRequests.forEach(r -> {
            DateTime requestDate = uuidToLastRecrawlRequest.get(r.getLeft());
            if (requestDate == null || requestDate.isBefore(r.getRight())) {
                uuidToLastRecrawlRequest.put(r.getLeft(), r.getRight());
            }
        });
        return Pair.of(uuidToLastRecrawlRequest, newRequestForHostIsAllowed(recrawlRequests));
    }

    public DateTime getLastRequestTime(WebmasterHostId hostId, UUID sitemapId) {
        return sitemapRecrawlRequestsYDao.getLastRequestTime(hostId, sitemapId);
    }

    private int countRequestsLastMonth(WebmasterHostId hostId) {
        return SitemapRecrawlQuotaUtil.getQuotaUsed(sitemapRecrawlRequestsYDao.getListOfRequestTimes(hostId), DAYS_IN_MONTH);
    }

    private boolean isRecrawlLimitNotExceeded(WebmasterHostId hostId) {
        return (countRequestsLastMonth(hostId) < RECRAWL_SITEMAP_QUOTA_PER_MONTH);
    }

    public boolean newRequestForHostIsAllowed(WebmasterHostId hostId) {
        return isRecrawlLimitNotExceeded(hostId);
    }

    public boolean newRequestForHostIsAllowed(List<Pair<UUID, DateTime>> recrawlRequests) {
        return SitemapRecrawlQuotaUtil.quotaIsRemain(recrawlRequests, RECRAWL_SITEMAP_QUOTA_PER_MONTH, DAYS_IN_MONTH);
    }

    public SitemapRecrawlQuotaUtil.QuotaUsage getHostRecrawlQuotaInfo(WebmasterHostId hostId) {
        List<DateTime> dates = sitemapRecrawlRequestsYDao.getListOfRequestTimes(hostId);
        return SitemapRecrawlQuotaUtil.computeQuotaInfo(dates, RECRAWL_SITEMAP_QUOTA_PER_MONTH, DAYS_IN_MONTH);
    }

    public Collection<SitemapRecrawlRequest> getLatestHostRequests(WebmasterHostId hostId) {
        // получим все запросы
        List<SitemapRecrawlRequest> hostRequests = sitemapRecrawlRequestsYDao.getRequestTimesExt(hostId);

        // оставим только самые последние даты
        Map<UUID, SitemapRecrawlRequest> uuidToLatestRequest = new HashMap<>(hostRequests.size());
        hostRequests.forEach(r -> {
            var latestReq = uuidToLatestRequest.get(r.getSitemapId());
            if (latestReq == null) {
                uuidToLatestRequest.put(r.getSitemapId(), r);
                return;
            }

            var latestReqTime = latestReq.getRequestDate();
            if (latestReqTime.isBefore(r.getRequestDate())) {
                uuidToLatestRequest.put(r.getSitemapId(), r);
            }
        });

        return uuidToLatestRequest.values();
    }

}
