package ru.yandex.webmaster3.storage.sitemap.dao;

import java.util.List;
import java.util.UUID;
import java.util.function.Consumer;

import org.apache.commons.lang3.mutable.MutableObject;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;

/**
 * Created by leonidrom on 23/05/2017.
 * С учетом того, что количество запросов на переобход сайтмепов меньше 100 в месяц, можно не заморачиваться с пагинацией
 */
@Repository
public class SitemapRecrawlRequestsYDao extends AbstractYDao {

    public SitemapRecrawlRequestsYDao() {
        super(PREFIX_SITEMAP, "sitemap_recrawl_requests");
    }

    public List<DateTime> getListOfRequestTimes(WebmasterHostId hostId) {
        return select(F.REQUEST_TIME).where(F.HOST_ID.eq(hostId)).queryForList();
    }

    public List<Pair<UUID, DateTime>> getRequestTimes(WebmasterHostId hostId) {
        return select(F.SITEMAP_ID.combine(F.REQUEST_TIME, Pair::of)).where(F.HOST_ID.eq(hostId)).queryForList();
    }

    public List<SitemapRecrawlRequest> getRequestTimesExt(WebmasterHostId hostId){
        return select(MAPPER).where(F.HOST_ID.eq(hostId)).queryForList();
    }

    public DateTime getLastRequestTime(WebmasterHostId hostId, UUID sitemapId) {
        return select(F.REQUEST_TIME)
                .where(F.HOST_ID.eq(hostId))
                .and(F.SITEMAP_ID.eq(sitemapId))
                .limit(1)
                .queryOne();
    }

    public void storeRequest(WebmasterHostId hostId, UUID sitemapId, UUID parentSitemapId, String sitemapUrl, DateTime requestTime) {
        upsert(
                F.HOST_ID.value(hostId),
                F.SITEMAP_ID.value(sitemapId),
                F.PARENT_SITEMAP_ID.value(parentSitemapId),
                F.SITEMAP_URL.value(sitemapUrl),
                F.REQUEST_TIME.value(requestTime)
        ).execute();
    }

    public void forEachDistinctHost(Consumer<WebmasterHostId> consumer) {
        MutableObject<WebmasterHostId> lastHostId = new MutableObject<>();
        streamReader(F.HOST_ID, hostId -> {
            if (!hostId.equals(lastHostId.getValue())) {
                if (lastHostId.getValue() != null) {
                    consumer.accept(lastHostId.getValue());
                }
                lastHostId.setValue(hostId);
            }
        }, true);
        if (lastHostId.getValue() != null) {
            consumer.accept(lastHostId.getValue());
        }
    }

    private interface F {
        Field<WebmasterHostId> HOST_ID = Fields.hostIdField("host_id");
        Field<UUID> SITEMAP_ID = Fields.uuidField("sitemap_id");
        Field<String> SITEMAP_URL = Fields.stringField("sitemap_url");
        Field<DateTime> REQUEST_TIME = Fields.jodaDateTimeField("request_time");
        Field<UUID> PARENT_SITEMAP_ID = Fields.uuidField("parent_sitemap_id").makeOptional();
    }

    private static final DataMapper<SitemapRecrawlRequest> MAPPER = DataMapper.create(
            F.SITEMAP_ID, F.PARENT_SITEMAP_ID, F.REQUEST_TIME, SitemapRecrawlRequest::new
    );
}
