package ru.yandex.webmaster3.storage.spam;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.Select;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;

@Repository
public class SpamHostsYDao extends AbstractYDao {
    private static final String TABLE_NAME = "spam_hosts";
    private static final Duration TTL = Duration.standardDays(2);

    public SpamHostsYDao() {
        super(PREFIX_INTERNAL, TABLE_NAME);
    }

    public boolean hasHost(WebmasterHostId hostId) {
        return !select(F.HOST_ID)
                .where(F.HOST_ID.eq(hostId))
                .and(F.UPDATE_DATE.gt(Instant.now().minus(TTL)))
                .limit(1)
                .queryForList().isEmpty();
    }

    public Set<WebmasterHostId> getSpamHosts(Collection<WebmasterHostId> hostIds) {
        List<Select<WebmasterHostId>> statements = hostIds.stream().map(pair -> select(F.HOST_ID)
                        .where(F.HOST_ID.eq(pair))
                        .and(F.UPDATE_DATE.gt(Instant.now().minus(TTL)))
                        .getStatement())
                .collect(Collectors.toList());

        return new HashSet<>(asyncExecute(statements, F.HOST_ID));
    }

    public void insertHost(WebmasterHostId hostId) {
        upsert(F.HOST_ID.value(hostId),
                F.UPDATE_DATE.value(Instant.now()))
                .execute();
    }

    public void getAllHosts(Consumer<WebmasterHostId> consumer) {
        streamReader(F.HOST_ID, consumer);
    }

    private static class F {
        static final Field<WebmasterHostId> HOST_ID = Fields.hostIdField("host_id");
        static final Field<Instant> UPDATE_DATE = Fields.jodaInstantField("expire_after");
    }
}
