package ru.yandex.webmaster3.storage.sprav;

import java.util.List;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.ValueDataMapper;

/**
 * Created by kravchenko99 on 04.03.21.
 * TTL по полю ADD_TIME
 */
@Repository
public class CompanyProfileInfoYDao extends AbstractYDao {
    private static final DataMapper<CompanyProfileInfo> MAPPER = DataMapper.create(
            F.HOST_ID, F.TYCOON_ID, F.PERMALINK, F.PERCENTAGE, F.PERCENTAGE_DETAIL, F.FILLED, F.TO_FILL, F.URL, F.NAME,
            F.ADDRESS, F.POPULARITY, CompanyProfileInfo::new);
    private static final ValueDataMapper<Pair<CompanyProfileInfo, DateTime>> VALUE_DATA_MAPPER =
            ValueDataMapper.create2(
                    Pair.of(F.ADD_TIME, Pair::getValue),
                    Pair.of(F.HOST_ID, x -> x.getKey().getHostId()),
                    Pair.of(F.TYCOON_ID, x -> x.getKey().getTycoonId()),
                    Pair.of(F.PERMALINK, x -> x.getKey().getPermalink()),
                    Pair.of(F.PERCENTAGE, x -> x.getKey().getPercentage()),
                    Pair.of(F.PERCENTAGE_DETAIL, x -> x.getKey().getPercentageDetail()),
                    Pair.of(F.FILLED, x -> x.getKey().getFilled()),
                    Pair.of(F.TO_FILL, x -> x.getKey().getToFill()),
                    Pair.of(F.URL, x -> x.getKey().getUrl()),
                    Pair.of(F.NAME, x -> x.getKey().getName()),
                    Pair.of(F.ADDRESS, x -> x.getKey().getAddress()),
                    Pair.of(F.POPULARITY, x -> x.getKey().getPopularity())
            );
    //если менять кол-во MAX_COMPANIES_PER_HOST на больше чем 1000 то стоит менять на queryForList(...) в list
    private static final int MAX_COMPANIES_PER_HOST = 1000;
    private static final String TABLE_NAME = "company_profile_info";

    public CompanyProfileInfoYDao() {
        super(PREFIX_SPRAV, TABLE_NAME);
    }

    public void insertBatch(List<CompanyProfileInfo> infos, DateTime addTime) {
        List<Pair<CompanyProfileInfo, DateTime>> items =
                infos.stream().map(v -> Pair.of(v, addTime)).collect(Collectors.toList());
        batchInsert(VALUE_DATA_MAPPER, items).execute();
    }

    public void insertBatch(List<Pair<DateTime, CompanyProfileInfo>> infos) {
        List<Pair<CompanyProfileInfo, DateTime>> items =
                infos.stream().map(v -> Pair.of(v.getValue(), v.getKey())).collect(Collectors.toList());
        batchInsert(VALUE_DATA_MAPPER, items).execute();
    }

    public List<CompanyProfileInfo> list(WebmasterHostId hostId) {
        //если менять кол-во MAX_COMPANIES_PER_HOST на больше чем 1000 то стоит менять на queryForList(...)
        return select(MAPPER)
                .where(F.HOST_ID.eq(hostId))
                .order(F.TYCOON_ID.asc())
                .limit(MAX_COMPANIES_PER_HOST)
                .queryForList();
    }

    public void forEach(Consumer<Pair<DateTime, CompanyProfileInfo>> consumer) {
        streamReader(DataMapper.create(F.ADD_TIME, MAPPER, Pair::of), consumer);
    }

    private interface F {
        Field<WebmasterHostId> HOST_ID = Fields.hostIdField("host_id");
        Field<Long> TYCOON_ID = Fields.longField("tycoon_id");
        Field<Long> PERMALINK = Fields.longField("permalink");
        Field<Double> PERCENTAGE = Fields.doubleField("percentage");
        Field<FilledPercentageDetail> PERCENTAGE_DETAIL = Fields.jsonField2("percentage_detail",
                FilledPercentageDetail.class);
        Field<List<String>> FILLED = Fields.jsonField2("filled", JsonMapping.STRING_LIST_REFERENCE);
        Field<List<String>> TO_FILL = Fields.jsonField2("to_fill", JsonMapping.STRING_LIST_REFERENCE);
        Field<String> URL = Fields.stringField("url");
        Field<DateTime> ADD_TIME = Fields.jodaDateTimeField("add_time");
        Field<String> NAME = Fields.stringField("name_ru");
        Field<String> ADDRESS = Fields.stringField("address");
        Field<Double> POPULARITY = Fields.doubleField("popularity");
    }
}
