package ru.yandex.webmaster3.storage.takeout;

import java.util.UUID;

import javax.annotation.Nullable;

import org.joda.time.DateTime;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.util.enums.EnumResolver;
import ru.yandex.webmaster3.storage.user.UserTakeoutRequest;
import ru.yandex.webmaster3.storage.user.UserTakeoutRequestStatus;
import ru.yandex.webmaster3.core.user.takeout.UserTakeoutRequestType;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;

@Repository
public class UserTakeoutRequestsYDao extends AbstractYDao {
    private static final String TABLE_NAME = "user_takeout_requests";

    public UserTakeoutRequestsYDao() {
        super(PREFIX_INTERNAL, TABLE_NAME);
    }

    public void saveRequest(UserTakeoutRequest request) {
        upsert(
                F.REQUEST_ID.value(request.getRequestId()),
                F.STATUS.value(request.getStatus()),
                F.DATA.value(request.getData()),
                F.LAST_UPDATE.value(request.getLastUpdate()),
                F.USER_ID.value(request.getUserId()),
                F.TYPE.value(request.getType())
        ).execute();
    }

    public void markAsDone(UUID requestId, String data) {
        update(
                F.STATUS.set(UserTakeoutRequestStatus.DONE),
                F.LAST_UPDATE.set(DateTime.now()),
                F.DATA.set(data)
        ).where(F.REQUEST_ID.eq(requestId)).execute();
    }

    public void markAsDone(UUID requestId) {
        update(
                F.STATUS.set(UserTakeoutRequestStatus.DONE),
                F.LAST_UPDATE.set(DateTime.now())
        ).where(F.REQUEST_ID.eq(requestId)).execute();
    }

    public void markAsFailed(UUID requestId) {
        update(
                F.STATUS.set(UserTakeoutRequestStatus.FAILED),
                F.LAST_UPDATE.set(DateTime.now())
        ).where(F.REQUEST_ID.eq(requestId)).execute();
    }

    @Nullable
    public UserTakeoutRequest getRequest(UUID requestId) {
        return select(MAPPER)
                .where(F.REQUEST_ID.eq(requestId))
                .queryOne();
    }

    @Nullable
    public UserTakeoutRequest getRequest(long userId) {
        return select(MAPPER)
                .where(F.USER_ID.eq(userId))
                .queryOne();
    }

    private static final DataMapper<UserTakeoutRequest> MAPPER = DataMapper.create(
            F.REQUEST_ID, F.USER_ID, F.TYPE, F.STATUS, F.DATA, F.LAST_UPDATE, UserTakeoutRequest::new
    );

    private static final class F {
        static final Field<UUID> REQUEST_ID = Fields.uuidField("request_id");
        static final Field<Long> USER_ID = Fields.longField("user_id");
        static final Field<UserTakeoutRequestType> TYPE = Fields.intEnumField("type", UserTakeoutRequestType.R).withDefault(UserTakeoutRequestType.GET);
        static final Field<UserTakeoutRequestStatus> STATUS = Fields.intEnumField("status", UserTakeoutRequestStatus.R);
        static final Field<String> DATA = Fields.stringField("data").makeOptional();
        static final Field<DateTime> LAST_UPDATE = Fields.jodaDateTimeField("last_update");
    }

    public enum RequestType {
        GET,
        DELETE,
        ;

        public static final EnumResolver<RequestType> R = EnumResolver.er(RequestType.class);
    }
}
