package ru.yandex.webmaster3.storage.toloka;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.TimeUnit;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.http.HttpHeaders;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.message.AbstractHttpMessage;
import org.apache.http.util.EntityUtils;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.http.HttpConstants;
import ru.yandex.webmaster3.core.metrics.externals.AbstractExternalAPIService;
import ru.yandex.webmaster3.core.metrics.externals.ExternalDependencyMethod;
import ru.yandex.webmaster3.core.util.JavaMethodWitness;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.storage.toloka.model.TolokaAggregatedSolution;
import ru.yandex.webmaster3.storage.toloka.model.TolokaOperation;
import ru.yandex.webmaster3.storage.toloka.model.TolokaTask;
import ru.yandex.webmaster3.storage.toloka.model.TolokaTaskSuite;

/**
 * Created by Oleg Bazdyrev on 28/04/2020.
 */
@Slf4j
@Service("tolokaService")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class TolokaService extends AbstractExternalAPIService {

    private static final int SOCKET_TIMEOUT_MS = (int) Duration.standardMinutes(5L).getMillis();
    private static final int CONNECT_TIMEOUT_MS = HttpConstants.DEFAULT_CONNECT_TIMEOUT;

    @Value("https://sandbox.toloka.yandex.ru/api/v1")
    private String apiBaseUrl;
    @Value("TODO")
    private String authToken;

    private CloseableHttpClient httpClient;

    public void init() {
        RequestConfig requestConfig = RequestConfig.custom()
                .setSocketTimeout(SOCKET_TIMEOUT_MS)
                .setConnectTimeout(CONNECT_TIMEOUT_MS)
                .build();

        httpClient = HttpClients.custom()
                .setDefaultRequestConfig(requestConfig)
                .setConnectionTimeToLive(30, TimeUnit.SECONDS)
                .build();
    }

    @ExternalDependencyMethod("post-tasks")
    public Void postTasks(List<TolokaTask> tasks) throws Exception {
        return trackQuery(new JavaMethodWitness() {
                          }, ALL_ERRORS_INTERNAL, () -> {
                    HttpPost post = new HttpPost(new URIBuilder(apiBaseUrl + "/tasks")
                            .addParameter("allow_defaults", Boolean.TRUE.toString())
                            .addParameter("open_pool", Boolean.FALSE.toString())
                            .addParameter("async_mode", Boolean.FALSE.toString())
                            .toString());

                    addCommonHeaders(post);
                    post.setEntity(new StringEntity(JsonMapping.writeValueAsString(tasks), ContentType.APPLICATION_JSON));
                    log.info("Uploading {} tasks to Toloka", tasks.size());

                    try (CloseableHttpResponse response = httpClient.execute(post)) {
                        checkStatusCodeAndGetContent(response);
                        log.info("Successfully posted {} tasks to Toloka", tasks.size());
                        return null;
                    } catch (Exception e) {
                        throw new TolokaException("Exception when posting tasks to Toloka", e);
                    }
                }
        );
    }

    @ExternalDependencyMethod("post-task-suites")
    public Void postTaskSuites(List<TolokaTaskSuite> taskSuites) throws Exception {
        return trackQuery(new JavaMethodWitness() {
                          }, ALL_ERRORS_INTERNAL, () -> {
                    HttpPost post = new HttpPost(new URIBuilder(apiBaseUrl + "/task-suites")
                            .addParameter("allow_defaults", Boolean.TRUE.toString())
                            .addParameter("open_pool", Boolean.FALSE.toString())
                            .addParameter("async_mode", Boolean.FALSE.toString())
                            .toString());

                    addCommonHeaders(post);
                    post.setEntity(new StringEntity(JsonMapping.writeValueAsString(taskSuites), ContentType.APPLICATION_JSON));
                    log.info("Uploading {} task suites to Toloka", taskSuites.size());

                    try (CloseableHttpResponse response = httpClient.execute(post)) {
                        checkStatusCodeAndGetContent(response);
                        log.info("Successfully posted {} task suites to Toloka", taskSuites.size());
                        return null;
                    } catch (Exception e) {
                        throw new TolokaException("Exception when posting task suites to Toloka", e);
                    }
                }
        );
    }

    /**
     * Получение заданий в пуле (до 100к заданий)
     * @param poolId
     * @return
     */
    @ExternalDependencyMethod("get-tasks")
    public List<TolokaTask> getTasks(String poolId) throws Exception {
        return trackQuery(new JavaMethodWitness() {
            }, ALL_ERRORS_INTERNAL, () -> {

            log.info("Requesting tasks for pool {}", poolId);
            List<TolokaTask> result = new ArrayList<>();
            String lastTaskId = "";
            Pair<List<TolokaTask>, Boolean> tasks;
            do {
                tasks = getTasks(poolId, lastTaskId);
                result.addAll(tasks.getLeft());
                lastTaskId = result.isEmpty() ? "" : result.get(result.size() - 1).getId();
            } while (Boolean.TRUE.equals(tasks.getRight()));

            log.info("Found {} total tasks in pool {}", result.size(), poolId);

            return result;
        });
    }

    private Pair<List<TolokaTask>, Boolean> getTasks(String poolId, String fromId) throws Exception {
        HttpGet get = new HttpGet(new URIBuilder(apiBaseUrl + "/tasks")
                .addParameter("pool_id", poolId)
                .addParameter("sort", "id")
                .addParameter("id_gt", fromId)
                .addParameter("limit", String.valueOf(1000))
                .build()
        );
        addCommonHeaders(get);

        try (CloseableHttpResponse response = httpClient.execute(get)) {
            String content = checkStatusCodeAndGetContent(response);
            JsonNode parsedNode = JsonMapping.readValue(content, JsonNode.class);
            List<TolokaTask> result = JsonMapping.OM.readValue(parsedNode.get("items").traverse(), TolokaTask.TYPE_REFERENCE);
            boolean hasMoreItems = parsedNode.get("has_more").asBoolean();

            log.info("Successfully get {} tasks from Toloka", result.size());

            return Pair.of(result, hasMoreItems);
        } catch (Exception e) {
            throw new TolokaException("Exception when getting tasks from Toloka", e);
        }
    }

    @ExternalDependencyMethod("aggregate-by-pool")
    public TolokaOperation aggregateByPool(String poolId, String skillId, Collection<String> fields) throws Exception {
        return trackQuery(new JavaMethodWitness() {
                          }, ALL_ERRORS_INTERNAL, () -> {
                    HttpPost post = new HttpPost(new URIBuilder(apiBaseUrl + "/aggregate-by-pool")
                            .addParameter("allow_defaults", Boolean.TRUE.toString())
                            .addParameter("open_pool", Boolean.FALSE.toString())
                            .addParameter("async_mode", Boolean.FALSE.toString())
                            .toString());

                    addCommonHeaders(post);

                    ObjectNode data = JsonMapping.OM.createObjectNode();
                    data.put("pool_id", poolId);
                    data.put("type", "WEIGHTED_DYNAMIC_OVERLAP");
                    data.put("answer_weight_skill_id", skillId);
                    data.set("fields", fields.stream().map(field -> JsonMapping.OM.createObjectNode().put("name", field))
                            .collect(JsonMapping.OM::createArrayNode, ArrayNode::add, ArrayNode::addAll)
                    );

                    post.setEntity(new StringEntity(JsonMapping.writeValueAsString(data), ContentType.APPLICATION_JSON));
                    log.info("Starting aggregating results for pool {} and skill {}", poolId, skillId);

                    try (CloseableHttpResponse response = httpClient.execute(post)) {
                        String content = checkStatusCodeAndGetContent(response);
                        TolokaOperation operation = JsonMapping.readValue(content, TolokaOperation.class);
                        log.info("Successfully starting aggregating task {}", operation.getId());
                        return operation;
                    } catch (Exception e) {
                        throw new TolokaException("Exception when posting task suites to Toloka", e);
                    }
                }
        );
    }

    @ExternalDependencyMethod("get-operation")
    public TolokaOperation getOperation(String operationId) throws Exception {
        return trackQuery(new JavaMethodWitness() {
                          }, ALL_ERRORS_INTERNAL, () -> {
                    HttpGet get = new HttpGet(new URIBuilder(apiBaseUrl + "/operations/" + operationId).toString());

                    get.setHeader(HttpHeaders.AUTHORIZATION, "OAuth " + authToken);
                    get.setHeader(HttpHeaders.CONTENT_TYPE, ContentType.APPLICATION_JSON.getMimeType());

                    log.info("Retrieving operation info {}", operationId);

                    try (CloseableHttpResponse response = httpClient.execute(get)) {
                        String content = checkStatusCodeAndGetContent(response);
                        TolokaOperation operation = JsonMapping.readValue(content, TolokaOperation.class);
                        log.info("Successfully starting aggregating task {}", operation.getId());
                        return operation;
                    } catch (Exception e) {
                        throw new TolokaException("Exception when posting task suites to Toloka", e);
                    }
                }
        );
    }

    @ExternalDependencyMethod("get-aggregated-solutions")
    public List<TolokaAggregatedSolution> getAggregatedSolutions(String operationId) throws Exception {
        return trackQuery(new JavaMethodWitness() {
                          }, ALL_ERRORS_INTERNAL, () -> {
                    log.info("Requesting solutions for operation {}", operationId);
                    List<TolokaAggregatedSolution> result = new ArrayList<>();
                    String lastTaskId = "";
                    Pair<List<TolokaAggregatedSolution>, Boolean> tasks;
                    do {
                        tasks = getAggregatedSolutions(operationId, lastTaskId);
                        result.addAll(tasks.getLeft());
                        lastTaskId = result.isEmpty() ? "" : result.get(result.size() - 1).getTaskId();
                    } while (Boolean.TRUE.equals(tasks.getRight()));

                    log.info("Found {} total solutions for operation {}", result.size(), operationId);
                    return result;
                }
        );
    }

    private Pair<List<TolokaAggregatedSolution>, Boolean> getAggregatedSolutions(String operationId, String fromId) throws Exception {
        HttpGet get = new HttpGet(new URIBuilder(apiBaseUrl + "/aggregated-solutions/" + operationId)
                .addParameter("limit", "1000")
                .addParameter("sort", "task_id")
                .addParameter("task_id_gt", fromId)
                .toString()
        );
        addCommonHeaders(get);

        try (CloseableHttpResponse response = httpClient.execute(get)) {
            String content = checkStatusCodeAndGetContent(response);
            JsonNode parsedNode = JsonMapping.readValue(content, JsonNode.class);
            List<TolokaAggregatedSolution> result = JsonMapping.OM.readValue(parsedNode.get("items").traverse(), TolokaAggregatedSolution.TYPE_REFERENCE);
            boolean hasMoreItems = parsedNode.get("has_more").asBoolean();

            log.info("Successfully get {} solutions from Toloka", result.size());

            return Pair.of(result, hasMoreItems);
        } catch (Exception e) {
            throw new TolokaException("Exception when getting solutions from Toloka", e);
        }
    }

    private void addCommonHeaders(AbstractHttpMessage message) {
        message.setHeader(HttpHeaders.AUTHORIZATION, "OAuth " + authToken);
        message.setHeader(HttpHeaders.CONTENT_TYPE, ContentType.APPLICATION_JSON.getMimeType());
    }

    private String checkStatusCodeAndGetContent(CloseableHttpResponse response) throws Exception {
        int statusCode = response.getStatusLine().getStatusCode();
        if ((statusCode / 100) != 2) {
            log.error("Toloka response: {}", EntityUtils.toString(response.getEntity()));
            throw new TolokaException("Toloka return status code " + statusCode);
        }
        String content = EntityUtils.toString(response.getEntity());
        return content;
    }

}
