package ru.yandex.webmaster3.storage.turbo.dao;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.tuple.Pair;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.turbo.model.TurboDisplaySettings;
import ru.yandex.webmaster3.core.turbo.model.TurboHostHeader;
import ru.yandex.webmaster3.core.turbo.model.advertising.AdvertisingSettings;
import ru.yandex.webmaster3.core.turbo.model.desktop.TurboDesktopColumn;
import ru.yandex.webmaster3.core.turbo.model.desktop.TurboDesktopSettings;
import ru.yandex.webmaster3.core.util.W3Collectors;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;

/**
 * Created by Oleg Bazdyrev on 15/02/2019.
 */
@Repository
public class TurboDesktopSettingsYDao extends AbstractYDao {

    public static final String TABLE_NAME = "turbo_desktop_settings";
    private static final DataMapper<TurboDesktopSettings> MAPPER = DataMapper.create(F.ENABLED, F.COLUMNS,
            F.HEADER, F.ADVERTISING, F.CSS, F.MINIFIED_CSS, F.AAB_ENABLED, F.ANTI_ITP_ENABLED, F.DISPLAY, TurboDesktopSettings::new);
    private static final DataMapper<Pair<String, TurboDesktopSettings>> PAIR_MAPPER = DataMapper.create(
            TurboDesktopSettingsYDao.F.DOMAIN, MAPPER, Pair::of);

    protected TurboDesktopSettingsYDao() {
        super(PREFIX_TURBO, TABLE_NAME);
    }

    public void updateSettings(String domain, TurboDesktopSettings settings) {
        upsert(F.DOMAIN.value(domain),
                F.ENABLED.value(settings.getEnabled()),
                F.COLUMNS.value(settings.getColumns()),
                F.HEADER.value(settings.getHeader()),
                F.ADVERTISING.value(settings.getAdvertisingSettingsList()),
                F.CSS.value(settings.getCss()),
                F.MINIFIED_CSS.value(settings.getMinifiedCss()),
                F.AAB_ENABLED.value(settings.getAabEnabled()),
                F.ANTI_ITP_ENABLED.value(settings.getAntiItpEnabled()),
                F.DISPLAY.value(settings.getDisplaySettings())
        ).execute();
    }

    public TurboDesktopSettings getSettings(String domain) {
        return select(MAPPER).where(F.DOMAIN.eq(domain)).queryOne();
    }

    public Map<String, TurboDesktopSettings> getSettings(Collection<String> domains) {
        return select(PAIR_MAPPER).where(F.DOMAIN.in(domains)).queryForList().stream().collect(W3Collectors.toHashMap());
    }

    private interface F {
        Field<String> DOMAIN = Fields.stringField("domain");
        Field<Boolean> ENABLED = Fields.boolField("enabled").makeOptional();
        Field<Set<TurboDesktopColumn>> COLUMNS = Fields.setField("columns", TurboDesktopColumn.SET_REFERENCE).makeOptional();
        Field<TurboHostHeader> HEADER = Fields.jsonField2("header", TurboHostHeader.class).makeOptional();
        Field<List<AdvertisingSettings>> ADVERTISING = Fields.jsonField2("advertising", AdvertisingSettings.LIST_REFERENCE).makeOptional();
        Field<String> CSS = Fields.stringField("css").makeOptional();
        Field<String> MINIFIED_CSS = Fields.stringField("minified_css").makeOptional();
        Field<Boolean> AAB_ENABLED = Fields.boolField("aab_enabled").makeOptional();
        Field<Boolean> ANTI_ITP_ENABLED = Fields.boolField("anti_itp_enabled").makeOptional();
        Field<TurboDisplaySettings> DISPLAY = Fields.jsonField2("display", TurboDisplaySettings.class).makeOptional();
    }
}
