package ru.yandex.webmaster3.storage.turbo.dao;

import java.util.List;
import java.util.function.Function;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import lombok.Value;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.turbo.model.TurboSampleData;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedStatistics;
import ru.yandex.webmaster3.core.turbo.model.menu.TurboMenuItem;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.storage.checklist.data.RealTimeSiteProblemInfo;
import ru.yandex.webmaster3.storage.turbo.dao.automorda.TurboAutoMordaStatus;
import ru.yandex.webmaster3.storage.turbo.dao.commerce.model.TurboListingsInfo;
import ru.yandex.webmaster3.storage.turbo.dao.scc.model.TurboSccPremoderationStatus;
import ru.yandex.webmaster3.storage.turbo.service.TurboDomainsStateService;
import ru.yandex.webmaster3.storage.turbo.service.preview.app.TurboAppReviewsInfo;
import ru.yandex.webmaster3.storage.util.clickhouse2.AbstractClickhouseDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHField;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHPrimitiveType;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHRow;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHTable;

/**
 * @author kravchenko99
 * @date 2/15/21
 */
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Component
public class TurboDomainsStateHelper {
    public static final CHTable TABLE = CHTable.builder()
            .database(AbstractClickhouseDao.DB_WEBMASTER3_TURBO)
            .sharded(false)
            .name("turbo_domains_state_%s")
            .partitionBy("toYYYYMM(date)")
            .keyField(F.DATE, CHPrimitiveType.Date)
            .keyField(F.DOMAIN, CHPrimitiveType.String)
            .field(F.RSS_FEEDS, CHPrimitiveType.String)
            .field(F.YML_FEEDS, CHPrimitiveType.String)
            .field(F.BANS, CHPrimitiveType.String)
            .field(F.AUTORELATED_SAMPLES, CHPrimitiveType.String)
            .field(F.AUTOMORDA_SAMPLES, CHPrimitiveType.String)
            .field(F.AUTOMORDA_STATUS, CHPrimitiveType.String)
            .field(F.APP_REVIEWS_INFO, CHPrimitiveType.String)
            .field(F.AUTOPARSER_SAMPLES, CHPrimitiveType.String)
            .field(F.COMMERCE_CATEGORIES, CHPrimitiveType.String)
            .field(F.EXPERIMENT, CHPrimitiveType.String)
            .field(F.LISTINGS_INFO, CHPrimitiveType.String)
            .field(F.MARKET_FEEDS, CHPrimitiveType.String)
            .field(F.PROBLEMS, CHPrimitiveType.String)
            .field(F.BANNED_SCC, CHPrimitiveType.String)
            .field(F.SHOP_STATES, CHPrimitiveType.String)
            .field(F.PREMODERATION_RESULT, CHPrimitiveType.String)
            .partsToThrowInsert(1024)
            .build();

    private final TurboDomainsStateService turboDomainsStateService;
    private static final List<String> ALL_FIELDS = TABLE.getFields().stream().filter(f -> !f.getName().equals(F.DATE)).map(CHField::getName).collect(Collectors.toList());
    private static final List<String> FEEDS_FIELDS = List.of(F.DOMAIN, F.YML_FEEDS, F.RSS_FEEDS);
    private static final List<String> PROBLEMS_FIELDS = List.of(F.DOMAIN, F.PROBLEMS);
    private static final List<String> BANNED_SCC_FIELDS = List.of(F.DOMAIN, F.BANNED_SCC);
    private static final List<String> SCC_FIELDS = List.of(F.DOMAIN, F.BANNED_SCC, F.PREMODERATION_RESULT);
    private static final List<String> AUTOMORDA_FIELDS = List.of(F.DOMAIN, F.AUTOMORDA_SAMPLES, F.AUTOMORDA_STATUS);
    private static final List<String> MARKET_FEEDS_FIELDS = List.of(F.DOMAIN, F.MARKET_FEEDS);
    private static final List<String> APP_REVIEW_FIELDS = List.of(F.DOMAIN, F.APP_REVIEWS_INFO);
    private static final List<String> AUTORELATED_FIELDS = List.of(F.DOMAIN, F.AUTORELATED_SAMPLES);
    private static final List<String> LISTING_INFO_FIELDS = List.of(F.DOMAIN, F.LISTINGS_INFO);
    private static final List<String> FEEDS_AND_AUTOMORDA_FIELDS = List.of(F.DOMAIN, F.YML_FEEDS, F.RSS_FEEDS, F.AUTOMORDA_SAMPLES, F.AUTOMORDA_STATUS);


    //public interface
    public TurboDomainsStateMapperAndFields getAllFields() {
        return TurboDomainsStateMapperAndFields.of(ALL_FIELDS, this::mapToDomainState);
    }

    public TurboDomainsStateMapperAndFields getFeedsFields() {
        return TurboDomainsStateMapperAndFields.of(FEEDS_FIELDS, this::feedsMapper);
    }

    public TurboDomainsStateMapperAndFields getFeedsAndAutomordaFields() {
        return TurboDomainsStateMapperAndFields.of(FEEDS_AND_AUTOMORDA_FIELDS, this::feedsAndAutomordaMapper);
    }

    public TurboDomainsStateMapperAndFields getProblemsFields() {
        return TurboDomainsStateMapperAndFields.of(PROBLEMS_FIELDS, this::problemsMapper);
    }

    public static TurboDomainsStateMapperAndFields getBannedSccFields() {
        return TurboDomainsStateMapperAndFields.of(BANNED_SCC_FIELDS, TurboDomainsStateHelper::bannedSccMapper);
    }

    public static TurboDomainsStateMapperAndFields getSccFields() {
        return TurboDomainsStateMapperAndFields.of(SCC_FIELDS, TurboDomainsStateHelper::sccMapper);
    }

    public static TurboDomainsStateMapperAndFields getAutomordaFields() {
        return TurboDomainsStateMapperAndFields.of(AUTOMORDA_FIELDS, TurboDomainsStateHelper::automordaMapper);
    }
    public static TurboDomainsStateMapperAndFields getMarketFeedsFields() {
        return TurboDomainsStateMapperAndFields.of(MARKET_FEEDS_FIELDS, TurboDomainsStateHelper::marketFeedsMapper);
    }

    public static TurboDomainsStateMapperAndFields getAppReviewFields() {
        return TurboDomainsStateMapperAndFields.of(APP_REVIEW_FIELDS, TurboDomainsStateHelper::appReviewMapper);
    }

    public static TurboDomainsStateMapperAndFields getAutorelatedFields() {
        return TurboDomainsStateMapperAndFields.of(AUTORELATED_FIELDS, TurboDomainsStateHelper::autorelatedMapper);
    }

    public static TurboDomainsStateMapperAndFields getListingInfoFields() {
        return TurboDomainsStateMapperAndFields.of(LISTING_INFO_FIELDS, TurboDomainsStateHelper::listingMapper);
    }


    //mappers
    // мапит все поля
    private TurboDomainsStateService.TurboDomainState mapToDomainState(CHRow row) {
        String domain = getDomain(row);
        return TurboDomainsStateService.TurboDomainState.builder()
                .domain(domain)
                .appReviewsInfo(getAppReviewsInfo(row))
                .automordaSamples(getAutomordaSamples(row))
                .autoMordaStatus(getAutoMordaStatus(row))
                .autoparserSamples(getAutoparserSamples(row))
                .autorelatedSamples(getAutorelatedSamples(row))
                .bans(getBans(row))
                .commerceCategories(getCommerceCategories(row))
                .experiment(getExperiment(row))
                .listingsInfo(getListingsInfo(row))
                .rssFeeds(getRssFeeds(row))
                .ymlFeeds(getYmlFeeds(row))
                .marketFeeds(getMarketFeeds(row))
                .premoderationResult(getPremoderationResult(row))
                .bannedScc(getBannedScc(row))
                .shopState(getShopState(row))
                .problems(getProblems(row, domain))
                .build();
    }

    // мапит только фиды
    private TurboDomainsStateService.TurboDomainState feedsMapper(CHRow row) {
        final String domain = getDomain(row);
        return TurboDomainsStateService.TurboDomainState.builder()
                .domain(domain)
                .rssFeeds(getRssFeeds(row))
                .ymlFeeds(getYmlFeeds(row))
                .build();
    }

    private TurboDomainsStateService.TurboDomainState problemsMapper(CHRow row) {
        final String domain = getDomain(row);
        return TurboDomainsStateService.TurboDomainState.builder()
                .domain(domain)
                .problems(getProblems(row, domain))
                .build();
    }

    private static TurboDomainsStateService.TurboDomainState bannedSccMapper(CHRow row) {
        final String domain = getDomain(row);
        return TurboDomainsStateService.TurboDomainState.builder()
                .domain(domain)
                .bannedScc(getBannedScc(row))
                .build();
    }

    private static TurboDomainsStateService.TurboDomainState sccMapper(CHRow row) {
        final String domain = getDomain(row);
        return TurboDomainsStateService.TurboDomainState.builder()
                .domain(domain)
                .bannedScc(getBannedScc(row))
                .premoderationResult(getPremoderationResult(row))
                .build();
    }

    private static TurboDomainsStateService.TurboDomainState automordaMapper(CHRow row) {
        final String domain = getDomain(row);
        return TurboDomainsStateService.TurboDomainState.builder()
                .domain(domain)
                .automordaSamples(getAutomordaSamples(row))
                .autoMordaStatus(getAutoMordaStatus(row))
                .build();
    }

    private static TurboDomainsStateService.TurboDomainState marketFeedsMapper(CHRow row) {
        final String domain = getDomain(row);
        return TurboDomainsStateService.TurboDomainState.builder()
                .domain(domain)
                .marketFeeds(getMarketFeeds(row))
                .build();
    }

    private static TurboDomainsStateService.TurboDomainState appReviewMapper(CHRow row) {
        final String domain = getDomain(row);
        return TurboDomainsStateService.TurboDomainState.builder()
                .domain(domain)
                .appReviewsInfo(getAppReviewsInfo(row))
                .build();
    }

    private static TurboDomainsStateService.TurboDomainState autorelatedMapper(CHRow row) {
        final String domain = getDomain(row);
        return TurboDomainsStateService.TurboDomainState.builder()
                .domain(domain)
                .autorelatedSamples(getAutorelatedSamples(row))
                .build();
    }

    private static TurboDomainsStateService.TurboDomainState listingMapper(CHRow row) {
        final String domain = getDomain(row);
        return TurboDomainsStateService.TurboDomainState.builder()
                .domain(domain)
                .listingsInfo(getListingsInfo(row))
                .build();
    }

    private TurboDomainsStateService.TurboDomainState feedsAndAutomordaMapper(CHRow row) {
        final String domain = getDomain(row);
        return TurboDomainsStateService.TurboDomainState.builder()
                .domain(domain)
                .rssFeeds(getRssFeeds(row))
                .ymlFeeds(getYmlFeeds(row))
                .autoMordaStatus(getAutoMordaStatus(row))
                .automordaSamples(getAutomordaSamples(row))
                .build();
    }

    ////
    @NotNull
    private List<RealTimeSiteProblemInfo> getProblems(CHRow row, String domain) {
        return JsonMapping.readValue(row.getString(F.PROBLEMS), TurboDomainsStateService.TurboDomainState.PROBLEMS_REFERENCE).entrySet().stream()
                .map(entry -> turboDomainsStateService.createProblemInfo(domain, entry.getKey(), entry.getValue()))
                .collect(Collectors.toList());
    }

    @NotNull
    private List<TurboFeedStatistics> getYmlFeeds(CHRow row) {
        return JsonMapping.readValue(row.getString(F.YML_FEEDS), TurboDomainsStateService.TurboFeedRawState.LIST_REFERENCE).stream()
                .map(turboDomainsStateService::feedStatisticsFromRawState).collect(Collectors.toList());
    }

    private static String getDomain(CHRow row) {
        return row.getString(F.DOMAIN);
    }

    @NotNull
    private List<TurboFeedStatistics> getRssFeeds(CHRow row) {
        return JsonMapping.readValue(row.getString(F.RSS_FEEDS), TurboDomainsStateService.TurboFeedRawState.LIST_REFERENCE).stream()
                .map(turboDomainsStateService::feedStatisticsFromRawState).collect(Collectors.toList());
    }

    private static TurboDomainsStateService.TurboShopState getShopState(CHRow row) {
        return JsonMapping.readValue(row.getString(F.SHOP_STATES), TurboDomainsStateService.TurboShopState.TYPE_REFERENCE);
    }

    private static TurboDomainsStateService.TurboBannedScc getBannedScc(CHRow row) {
        return JsonMapping.readValue(row.getString(F.BANNED_SCC), TurboDomainsStateService.TurboBannedScc.TYPE_REFERENCE);
    }

    private static TurboSccPremoderationStatus getPremoderationResult(CHRow row) {
        return JsonMapping.readValue(row.getString(F.PREMODERATION_RESULT), TurboSccPremoderationStatus.TYPE_REFERENCE);
    }

    private static List<TurboDomainsStateService.TurboMarketFeed> getMarketFeeds(CHRow row) {
        return JsonMapping.readValue(row.getString(F.MARKET_FEEDS), TurboDomainsStateService.TurboMarketFeed.LIST_REFERENCE);
    }

    private static TurboListingsInfo getListingsInfo(CHRow row) {
        return JsonMapping.readValue(row.getString(F.LISTINGS_INFO), TurboListingsInfo.class);
    }

    private static String getExperiment(CHRow row) {
        return row.getString(F.EXPERIMENT);
    }

    private static List<TurboMenuItem> getCommerceCategories(CHRow row) {
        return JsonMapping.readValue(row.getString(F.COMMERCE_CATEGORIES), TurboMenuItem.LIST_REFERENCE);
    }

    private static List<TurboDomainsStateService.TurboBanRawInfo> getBans(CHRow row) {
        return JsonMapping.readValue(row.getString(F.BANS), TurboDomainsStateService.TurboBanRawInfo.LIST_REFERENCE);
    }

    private static List<TurboSampleData> getAutorelatedSamples(CHRow row) {
        return JsonMapping.readValue(row.getString(F.AUTORELATED_SAMPLES), TurboSampleData.TYPE_REFERENCE);
    }

    private static List<TurboSampleData> getAutoparserSamples(CHRow row) {
        return JsonMapping.readValue(row.getString(F.AUTOPARSER_SAMPLES), TurboSampleData.TYPE_REFERENCE);
    }

    private static TurboAutoMordaStatus getAutoMordaStatus(CHRow row) {
        return TurboAutoMordaStatus.fromTurboName(row.getString(F.AUTOMORDA_STATUS));
    }

    private static List<TurboSampleData> getAutomordaSamples(CHRow row) {
        return JsonMapping.readValue(row.getString(F.AUTOMORDA_SAMPLES), TurboSampleData.TYPE_REFERENCE);
    }

    private static TurboAppReviewsInfo getAppReviewsInfo(CHRow row) {
        return JsonMapping.readValue(row.getString(F.APP_REVIEWS_INFO), TurboAppReviewsInfo.class);
    }

    @Value
    public static class TurboDomainsStateMapperAndFields {
        List<String> fields;
        Function<CHRow, TurboDomainsStateService.TurboDomainState> mapper;

        public static TurboDomainsStateMapperAndFields of(List<String> fields, Function<CHRow, TurboDomainsStateService.TurboDomainState> mapper) {
            return new TurboDomainsStateMapperAndFields(fields, mapper);
        }
    }


    public interface F {
        String DATE = "date";
        String DOMAIN = "domain";
        String RSS_FEEDS = "rss_feeds";
        String YML_FEEDS = "yml_feeds";
        String BANS = "bans";
        String AUTORELATED_SAMPLES = "autorelated_samples";
        String AUTOMORDA_SAMPLES = "automorda_samples";
        String AUTOMORDA_STATUS = "automorda_status";
        String APP_REVIEWS_INFO = "app_reviws_info";
        String AUTOPARSER_SAMPLES = "autoparser_samples";
        String COMMERCE_CATEGORIES = "commerce_categories";
        String EXPERIMENT = "experiment";
        String LISTINGS_INFO = "listings_info";
        String MARKET_FEEDS = "market_feeds";
        String PREMODERATION_RESULT = "premoderation_result";
        String BANNED_SCC = "banned_scc";
        String SHOP_STATES = "shop_states";
        String PROBLEMS = "problems";
    }

}
