package ru.yandex.webmaster3.storage.turbo.dao.commerce.model;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.collect.Sets;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Value;
import org.apache.commons.collections4.CollectionUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import ru.yandex.webmaster3.core.util.json.JsonMapping;

/**
 * @author: ishalaru
 * DATE: 28.08.2019
 */
@Value
@Builder(toBuilder = true)
@AllArgsConstructor
@JsonIgnoreProperties(ignoreUnknown = true)
public class TurboListingsInfo {

    private static final Set<String> VERDICTS_FOR_TURNING_ON = Sets.newHashSet("OK", "WARN");

    @JsonProperty("host")
    String host;
    @JsonProperty("verdict")
    String verdict;
    @JsonProperty("samples")
    List<TurboAutoListingsSample> samples;
    @JsonProperty("errors")
    List<TurboAutoListingsError> errors;

    @Value
    @Builder
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class TurboAutoListingsSample {

        @JsonProperty("original_url")
        String originalUrl;
        @JsonProperty("turbo_url")
        String turboUrl;
        @JsonProperty("errors")
        List<TurboAutoListingsError> errors;

        public List<TurboAutoListingsError> getErrorsWithDetails() {
            if (CollectionUtils.isEmpty(errors)) {
                return Collections.emptyList();
            }
            List<TurboAutoListingsError> result = new ArrayList<>();
            for (TurboAutoListingsError error : errors) {
                error = TurboAutoListingsError.builder()
                        .code(error.getCode())
                        .details(error.getDetails() != null ? error.getDetails() : JsonMapping.OM.createObjectNode())
                        .isPublic(error.getIsPublic())
                        .severity(error.getSeverity())
                        .build();
                error.getDetails().put("original_url", originalUrl).put("turbo_url", turboUrl);
                result.add(error);
            }
            return result;
        }
    }

    @Value
    @Builder
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class TurboAutoListingsError {
        @JsonProperty("code")
        String code;
        @JsonProperty("details")
        ObjectNode details;
        @JsonProperty("severity")
        String severity;
        @JsonProperty("public")
        Boolean isPublic;

        @JsonIgnore
        public boolean isVisible() {
            return Boolean.TRUE.equals(isPublic);
        }
    }


    @JsonIgnore
    public boolean canBeTurnedOn() {
        return VERDICTS_FOR_TURNING_ON.contains(verdict);
    }
}
