package ru.yandex.webmaster3.storage.turbo.dao.statistics;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.turbo.model.TurboUrl;
import ru.yandex.webmaster3.core.turbo.model.error.TurboRawError;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedItemStatistics;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedStatistics;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedType;
import ru.yandex.webmaster3.storage.util.JsonDBMapping;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;

/**
 * DAO для статистики обхода турбо-фидов (без бесполезного host_id)
 * Будет хранить в себе данные только по отключенным фидам, все статусы включенных фидов см. в TurboDomainsStateCHDao
 */
@Repository
public class TurboFeedsStatistics2YDao extends AbstractYDao {

    private static final String TABLE_NAME = "turbo_feeds_statistics2";

    private static final ObjectMapper OM = JsonDBMapping.OM.copy()
            .enable(DeserializationFeature.READ_UNKNOWN_ENUM_VALUES_AS_NULL);
    private static final DataMapper<TurboFeedStatistics> MAPPER =
            DataMapper.create(F.TYPE, F.URL, F.ACTIVE, F.TURBO_URLS, F.UPDATE_DATE, F.DOWNLOAD_DATE, F.RAW_ERRORS,
                    F.STATS, F.HASH, TurboFeedStatistics::new);

    public TurboFeedsStatistics2YDao() {
        super(PREFIX_TURBO, TABLE_NAME);
    }

    public void updateStatistics(TurboFeedStatistics statistics) {
        execute(
                upsert(F.URL.value(statistics.getUrl()),
                        F.ACTIVE.value(statistics.isActive()),
                        F.TYPE.value(statistics.getType()),
                        F.TURBO_URLS.value(statistics.getTurboUrls()),
                        F.UPDATE_DATE.value(statistics.getUpdateDate()),
                        F.DOWNLOAD_DATE.value(statistics.getDownloadDate()),
                        F.RAW_ERRORS.value(statistics.getRawErrors()),
                        F.STATS.value(statistics.getStats()),
                        F.HASH.value(statistics.getHash()))
        );
    }

    public TurboFeedStatistics getStatistics(String feedUrl, boolean isActive) {
        return queryOne(select(MAPPER).where(F.URL.eq(feedUrl)).and(F.ACTIVE.eq(isActive)).getStatement(), MAPPER);
    }

    /**
     * Получение статусов обхода фидов (в карте виде url + active)
     */
    public Map<Pair<String, Boolean>, TurboFeedStatistics> getStatistics(Collection<String> feedUrls) {
        Preconditions.checkArgument(!CollectionUtils.isEmpty(feedUrls));

        return queryForList(select(MAPPER).where(F.URL.in(feedUrls)).getStatement(), MAPPER).stream()
                .collect(Collectors.toMap(tfs -> Pair.of(tfs.getUrl(), tfs.isActive()), Function.identity()));
    }

    public void deleteStatistics(String feedUrl) {
        execute(delete().where(F.URL.eq(feedUrl)).getStatement());
    }

    private interface F {
        Field<String> URL = Fields.stringField("url");
        Field<Boolean> ACTIVE = Fields.boolField("active");
        Field<TurboFeedType> TYPE = Fields.stringEnumField("type", TurboFeedType.R);
        Field<List<TurboUrl>> TURBO_URLS = Fields.jsonField2("turbo_urls", TurboUrl.TYPE_REFERENCE).makeOptional();
        Field<DateTime> UPDATE_DATE = Fields.jodaDateTimeField("update_date").makeOptional();
        Field<DateTime> DOWNLOAD_DATE = Fields.jodaDateTimeField("download_date").makeOptional();
        Field<List<TurboRawError>> RAW_ERRORS = Fields.jsonField2("raw_errors", TurboRawError.TYPE_REFERENCE, OM).makeOptional();
        Field<TurboFeedItemStatistics> STATS = Fields.jsonField2("stats", TurboFeedItemStatistics.class).makeOptional();
        Field<String> HASH = Fields.stringField("hash").makeOptional();
    }
}
