package ru.yandex.webmaster3.storage.turbo.dao.statistics;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.turbo.model.TurboSource;
import ru.yandex.webmaster3.core.turbo.model.TurboType;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.storage.clickhouse.LegacyMdbTableStorage;
import ru.yandex.webmaster3.storage.clickhouse.LocalClickhouseTableProvider;
import ru.yandex.webmaster3.storage.clickhouse.TableType;
import ru.yandex.webmaster3.storage.searchurl.samples.data.TurboSearchUrlSample;
import ru.yandex.webmaster3.storage.util.clickhouse2.AbstractClickhouseDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseServer;
import ru.yandex.webmaster3.storage.util.clickhouse2.MdbClickhouseServer;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.QueryBuilder;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.Statement;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.Where;

public class TurboSearchUrlSamplesCHDao  extends AbstractClickhouseDao {
    private static final long SOURCE_FLAGS_ALL          = ~0;

    private static final long SOURCE_FLAGS_CONTENT      = Arrays.stream(TurboSource.values())
        .filter(TurboSource::isNotYml)
        .map(TurboSource::value)
        .reduce(0, (a, b) -> a | b);

    private static final long SOURCE_FLAGS_ECOMMERCE    = Arrays.stream(TurboSource.values())
        .filter(TurboSource::isYml)
        .map(TurboSource::value)
        .reduce(0, (a, b) -> a | b);

    @Autowired
    private LegacyMdbTableStorage mdbTableStorage;
    @Autowired
    private MdbClickhouseServer legacyMdbClickhouseServer;

    @Override
    protected ClickhouseServer getClickhouseServer() {
        return legacyMdbClickhouseServer;
    }

    public Optional<Long> getSamplesCount(String domain, long filterSourceFlags) {
        LocalClickhouseTableProvider table = mdbTableStorage.getTable(TableType.TURBO_SEARCHURLS_SAMPLES);

        Where st = QueryBuilder
            .select()
            .countAll()
            .from(table.getTableName())
            .where(QueryBuilder.eq(F.DOMAIN, domain))
            .and(
                QueryBuilder.ne(
                    QueryBuilder.bitAnd(F.SOURCE_FLAGS, filterSourceFlags).toString(),
                    0
                )
            )
        ;

        return getClickhouseServer().queryOne(table.chContext(getClickhouseServer(), domain), st.toString(),
                chRow -> chRow.getLongUnsafe(0));
    }

    public Optional<Long> getSamplesCount(String domain) {
        return getSamplesCount(domain, SOURCE_FLAGS_ALL);
    }

    public Optional<Long> getSamplesCount(String domain, TurboType turboType) {
        switch(turboType) {
            case CONTENT:
                return getSamplesCount(domain, SOURCE_FLAGS_CONTENT);
            case ECOMMERCE:
                return getSamplesCount(domain, SOURCE_FLAGS_ECOMMERCE);
        }
        return Optional.of(0L);
    }

    public List<TurboSearchUrlSample> getSamples(String domain, long skip, long limit, long filterSourceFlags) {
        LocalClickhouseTableProvider table = mdbTableStorage.getTable(TableType.TURBO_SEARCHURLS_SAMPLES);

        Statement st = QueryBuilder
            .select(
                F.DOMAIN,
                F.URL,
                F.FEED,
                F.SOURCE,
                F.IS_SEARCHABLE
            )
            .from(table.getTableName())
            .where(QueryBuilder.eq(F.DOMAIN, domain))
            .and(
                QueryBuilder.ne(
                    QueryBuilder.bitAnd(F.SOURCE_FLAGS, filterSourceFlags).toString(),
                    0
                )
            )
            // почему-то лимиты реализованы только для интов, наверное, стоит переделать на лонги
            .limit((int) skip, (int) limit);

        return getClickhouseServer().queryAll(table.chContext(getClickhouseServer(), domain), st.toString(),
                chRow -> new TurboSearchUrlSample(chRow.getString(0), chRow.getString(1), chRow.getString(2),
                                chRow.getString(3), chRow.getIntUnsafe(4)));
    }

    public List<TurboSearchUrlSample> getSamples(String domain, long skip, long limit) {
        return getSamples(domain, skip, limit, SOURCE_FLAGS_ALL);
    }

    public List<TurboSearchUrlSample> getSamples(String domain, long skip, long limit, TurboType turboType) {
        switch(turboType) {
            case CONTENT:
                return getSamples(domain, skip, limit, SOURCE_FLAGS_CONTENT);
            case ECOMMERCE:
                return getSamples(domain, skip, limit, SOURCE_FLAGS_ECOMMERCE);
        }

        return Collections.emptyList();
    }

    public List<String> getSamples(String domain, String feedUrl, String turboUrl) {
        LocalClickhouseTableProvider table = mdbTableStorage.getTable(TableType.TURBO_SEARCHURLS_SAMPLES);

        Statement st = QueryBuilder
                .select(F.URL)
                .from(table.getTableName())
                .where(QueryBuilder.eq(F.DOMAIN, domain))
                .and(QueryBuilder.eq(F.FEED, feedUrl))
                .and(QueryBuilder.eq(F.URL, turboUrl));

        return getClickhouseServer().queryAll(table.chContext(getClickhouseServer(), domain), st.toString(),
                chRow -> chRow.getString(F.URL));
    }

    public boolean hasTurboVersion(String url) {
        String domain = WwwUtil.cutWWWAndM(IdUtils.urlToHostId(url));
        LocalClickhouseTableProvider table = mdbTableStorage.getTable(TableType.TURBO_SEARCHURLS_SAMPLES);

        Statement st = QueryBuilder
                .select(F.URL)
                .from(table.getTableName())
                .where(QueryBuilder.eq(F.DOMAIN, domain))
                .and(QueryBuilder.eq(F.URL, url));

        return getClickhouseServer().queryOne(table.chContext(getClickhouseServer(), domain), st.toString(),
                chRow -> chRow.getString(F.URL)).isPresent();
    }

    public interface F {
        String DOMAIN = "domain";
        String URL = "url";
        String FEED = "feed";
        String SOURCE = "source";
        String SOURCE_FLAGS = "source_flags";
        String IS_SEARCHABLE = "is_searchable";
    }
}
