package ru.yandex.webmaster3.storage.turbo.problems;

import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.collect.Maps;
import lombok.RequiredArgsConstructor;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.checklist.data.SiteProblemStorageType;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.data.WebmasterHostGeneration;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.storage.checklist.data.AbstractProblemInfo;
import ru.yandex.webmaster3.storage.checklist.service.SiteProblemsProvider;
import ru.yandex.webmaster3.storage.turbo.dao.TurboDomainsStateCHDao;
import ru.yandex.webmaster3.storage.turbo.service.TurboDomainsStateService;

/**
 * Created by Oleg Bazdyrev on 03/08/2020.
 */
@Service
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class TurboProblemsProvider implements SiteProblemsProvider {

    private static final Set<SiteProblemTypeEnum> SUPPORTED_PROBLEM_TYPES = Collections.unmodifiableSet(Stream.of(SiteProblemTypeEnum.values())
            .filter(type -> type.getStorageType() == SiteProblemStorageType.TURBO_PROBLEM)
            .filter(type -> !type.isDisabled())
            .collect(Collectors.toCollection(() -> EnumSet.noneOf(SiteProblemTypeEnum.class))));

    private final TurboDomainsStateCHDao turboDomainsStateCHDao;

    @Override
    public @NotNull List<? extends AbstractProblemInfo> listProblems(@NotNull WebmasterHostId hostId, @Nullable UUID generationId) {
        return turboDomainsStateCHDao.getDomainStateOnlyProblems(WwwUtil.cutWWWAndM(hostId)).getProblems();
    }

    @Override
    public Map<WebmasterHostId, List<? extends AbstractProblemInfo>> listProblems(List<WebmasterHostGeneration> hosts) {
        Map<WebmasterHostId, String> hostIdToDomain = hosts.stream()
                .map(WebmasterHostGeneration::getHostId)
                .map(hostId -> Maps.immutableEntry(hostId, WwwUtil.cutWWWAndM(hostId)))
                .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));


        // кроме проблем и домена остальные поля пусты
        Map<String, TurboDomainsStateService.TurboDomainState> problemsMap = turboDomainsStateCHDao.getDomainsStateOnlyProblems(hostIdToDomain.values());
        return hostIdToDomain.entrySet().stream()
                .map(e -> {
                    var hostId = e.getKey();
                    var domain = e.getValue();
                    TurboDomainsStateService.TurboDomainState turboDomainState = problemsMap.get(domain);
                    return Maps.immutableEntry(hostId, turboDomainState == null? null : turboDomainState.getProblems());
                })
                .filter(e -> e.getValue() != null)
                .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
    }

    @Override
    public @NotNull Set<SiteProblemTypeEnum> getSupportedProblemTypes() {
        return SUPPORTED_PROBLEM_TYPES;
    }
}
