package ru.yandex.webmaster3.storage.turbo.service.autoparser;

import java.util.ArrayList;
import java.util.List;

import lombok.RequiredArgsConstructor;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.turbo.model.autoparser.AutoparserToggleState;
import ru.yandex.webmaster3.core.turbo.model.autoparser.TurboAutoparsedHostInfo;
import ru.yandex.webmaster3.storage.host.CommonDataState;
import ru.yandex.webmaster3.storage.host.CommonDataType;
import ru.yandex.webmaster3.storage.settings.SettingsService;
import ru.yandex.webmaster3.storage.turbo.dao.autoparser.TurboAutoparsedYDao;

/**
 * Created by ifilippov5 on 19.06.18.
 */
@Service
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class TurboAutoparserInfoService {
    private static final Logger log = LoggerFactory.getLogger(TurboAutoparserInfoService.class);

    private final TurboAutoparsedYDao turboAutoparsedYDao;
    private final SettingsService settingsService;

    /**
     * Дата последнего импорта из yt
     */
    public DateTime getLastImportDate() {
        CommonDataState state = settingsService.getSettingCached(CommonDataType.TURBO_AUTOPARSED_HOSTS_LAST_IMPORT_DATE);
        return (state == null ? null : DateTime.parse(state.getValue()));
    }

    /**
     * Является ли инфа актуальной после последнего импорта
     */
    public boolean isHostInfoActual(TurboAutoparsedHostInfo info) {
        return isHostInfoActual(info, getLastImportDate());
    }

    public boolean isHostInfoActual(TurboAutoparsedHostInfo info, DateTime lastImportDate) {
        if (info == null || info.getUpdateDate() == null) {
            return false;
        }
        if (lastImportDate == null) {
            log.info("Any import not found");
            return false;
        }
        return lastImportDate.toString().equals(info.getUpdateDate().toString());
    }

    /**
     * Получает актуальное состояние галки автопарсера с учетом наследования
     *
     * @param host
     * @return
     * @
     */
    public AutoparserToggleState getAutoparseCheckBoxState(String host)  {
        DateTime lastImportDate = getLastImportDate();
        List<String> tlds = new ArrayList<>();
        tlds.add(host);
        tlds.addAll(TurboAutoparserInfoUtil.getParentsChain(host, ""));
        for (String tld : tlds) {
            TurboAutoparsedHostInfo info = turboAutoparsedYDao.getShort(tld);
            if (info == null || info.getUpdateDate() == null || !isHostInfoActual(info, lastImportDate)) {
                if (host.equals(tld)) {
                    // если искомый хост уже протух - дальше смотреть не будем
                    return AutoparserToggleState.ABSENT;
                }
                continue;
            }
            if (info.getCheckboxState() != AutoparserToggleState.INHERITS) {
                return info.getCheckboxState();
            }
        }
        return AutoparserToggleState.OFF;
    }

}
