package ru.yandex.webmaster3.storage.turbo.service.preview;

import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.node.ObjectNode;
import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.ObjectUtils;

import ru.yandex.webmaster3.core.turbo.model.TurboHostSettings;
import ru.yandex.webmaster3.core.turbo.model.advertising.AdvertisingSettings;
import ru.yandex.webmaster3.core.turbo.model.desktop.TurboDesktopColumn;
import ru.yandex.webmaster3.core.turbo.model.desktop.TurboDesktopSettings;
import ru.yandex.webmaster3.core.turbo.model.menu.TurboMenuItem;
import ru.yandex.webmaster3.core.turbo.model.search.TurboSearchSettings;
import ru.yandex.webmaster3.storage.util.JsonDBMapping;

/**
 * Created by Oleg Bazdyrev on 25/02/2019.
 */
@RequiredArgsConstructor
public class TurboDesktopSettingsData {

    private final boolean enabled;
    private final List<String> layout;
    private final String css;
    private final List<ObjectNode> ads;
    private final TurboHostSettingsData.Logo logo;
    private final List<ObjectNode> topMenu;
    private final boolean aabEnabled;
    private final boolean antiItpEnabled;

    @JsonProperty("enabled")
    public boolean isEnabled() {
        return enabled;
    }

    @JsonProperty("layout")
    public List<String> getLayout() {
        return layout;
    }

    @JsonProperty("css")
    public String getCss() {
        return css;
    }

    @JsonProperty("ads")
    public List<ObjectNode> getAds() {
        return ads;
    }

    @JsonProperty("logo")
    public TurboHostSettingsData.Logo getLogo() {
        return logo;
    }

    @JsonProperty("top_menu")
    public List<ObjectNode> getTopMenu() {
        return topMenu;
    }

    @JsonProperty("aab_enabled")
    public boolean isAabEnabled() {
        return aabEnabled;
    }

    @JsonProperty("anti_itp_enabled")
    public boolean isAntiItpEnabled() {
        return antiItpEnabled;
    }

    public static TurboDesktopSettingsData fromDesktopSettings(
            TurboHostSettings hostSettings, TurboDesktopSettings desktopSettings) {
        if (desktopSettings == null) {
            return null;
        }
        EnumSet<TurboDesktopColumn> columns = EnumSet.of(TurboDesktopColumn.MAIN);
        if (desktopSettings.getColumns() != null) {
            columns.addAll(desktopSettings.getColumns());
        }
        List<String> layout = columns.stream().map(TurboDesktopColumn::getDisplayName).collect(Collectors.toList());

        List<ObjectNode> topMenu = new ArrayList<>();
        if (hostSettings.getTopMenuSettings() != null) {
            hostSettings.getTopMenuSettings().stream().map((TurboMenuItem menuItem) -> {
                ObjectNode node = JsonDBMapping.OM.createObjectNode();
                node.put("url", menuItem.getUrl());
                node.put("text", menuItem.getLabel());
                return node;
            }).forEach(topMenu::add);
        }
        TurboSearchSettings searchSettings = hostSettings.getSearchSettings();
        if (searchSettings != null) {
            ObjectNode searchNode = JsonDBMapping.OM.createObjectNode();
            searchNode.put("url", searchSettings.getUrl());
            searchNode.put("charset", searchSettings.getCharset());
            searchNode.put("placeholder", searchSettings.getPlaceholder());
            topMenu.add(searchNode);
        }

        // TODO безумный костыль, INSTREAM должен быть и на десктопе
        List<ObjectNode> ads = desktopSettings.getAdvertisingSettingsList().stream()
                .map(AdvertisingSettings::toJson).collect(Collectors.toList());
        ads.addAll(hostSettings.getAdvertisingSettingsList().stream().filter(ad -> ad.getType().isInstream())
                .map(AdvertisingSettings::toJson).collect(Collectors.toList()));

        return new TurboDesktopSettingsData(
                Boolean.TRUE.equals(desktopSettings.getEnabled()),
                layout,
                ObjectUtils.firstNonNull(desktopSettings.getMinifiedCss(), desktopSettings.getCss()),
                ads,
                TurboHostSettingsData.Logo.fromTurboLogo(desktopSettings.getHeader() == null ? null : desktopSettings.getHeader().getLogoInfo()),
                topMenu,
                Boolean.TRUE.equals(desktopSettings.getAabEnabled()),
                Boolean.TRUE.equals(desktopSettings.getAntiItpEnabled())
                );
    }
}
