package ru.yandex.webmaster3.storage.turbo.service.preview.commerce;

import java.math.BigDecimal;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonProperty;
import lombok.AccessLevel;
import lombok.Builder;
import lombok.Value;

import ru.yandex.webmaster3.core.turbo.model.commerce.CurrencyEnum;
import ru.yandex.webmaster3.core.turbo.model.commerce.delivery.CourierDeliverySettings;
import ru.yandex.webmaster3.core.turbo.model.commerce.delivery.DeliverySection;
import ru.yandex.webmaster3.core.turbo.model.commerce.delivery.MailDeliverySettings;
import ru.yandex.webmaster3.core.turbo.model.commerce.delivery.PickupDeliverySettings;

/**
 * @author: ishalaru
 * DATE: 30.07.2019
 */
@Value
@Builder(access = AccessLevel.PRIVATE)
public class DeliveryInfo {
    @JsonProperty("mail")
    List<MailDeliveryInfo> mail;
    @JsonProperty("pickup")
    List<PickupDeliveryInfo> pickup;
    @JsonProperty("courier")
    List<CourierDeliveryInfo> courier;
    @JsonProperty("currency")
    CurrencyEnum currency;

    public static DeliveryInfo create(DeliverySection deliverySection) {
        if (deliverySection == null) {
            return null;
        }
        List<CourierDeliveryInfo> courierDeliveryInfos = List.of();
        if (deliverySection.getCourier() != null)
            courierDeliveryInfos = deliverySection.getCourier().stream().map(s -> CourierDeliveryInfo.create(deliverySection, s))
                    .collect(Collectors.toList());
        List<MailDeliveryInfo> mailDeliveryInfos = List.of();
        if (deliverySection.getMail() != null) {
            mailDeliveryInfos = deliverySection.getMail().stream().map(s -> MailDeliveryInfo.create(deliverySection, s))
                    .collect(Collectors.toList());
        }
        List<PickupDeliveryInfo> pickupDeliveryInfos = List.of();
        if (deliverySection.getPickup() != null) {
            pickupDeliveryInfos = deliverySection.getPickup().stream().map(s -> PickupDeliveryInfo.create(deliverySection, s))
                    .collect(Collectors.toList());
        }

        return DeliveryInfo.builder()
                .courier(courierDeliveryInfos)
                .mail(mailDeliveryInfos)
                .pickup(pickupDeliveryInfos)
                .currency(deliverySection.getCurrency())
                .build();
    }

    @Value
    @Builder(toBuilder = true)
    public static class MailDeliveryInfo {
        @JsonProperty("name")
        String name;
        @JsonProperty("min_delivery_period")
        int minDeliveryPeriod;
        @JsonProperty("max_delivery_period")
        int maxDeliveryPeriod;
        @JsonProperty("price")
        BigDecimal price;
        @JsonProperty("free_from")
        BigDecimal freeFrom;
        @JsonProperty("vat")
        String vat;
        @Deprecated(forRemoval = true)
        @JsonProperty("currency")
        CurrencyEnum currency;

        public static MailDeliveryInfo create(DeliverySection section, MailDeliverySettings other) {
            return MailDeliveryInfo.builder()
                    .name(other.getName())
                    .minDeliveryPeriod(other.getMinDeliveryPeriod())
                    .maxDeliveryPeriod(other.getMaxDeliveryPeriod())
                    .price(other.getPrice())
                    .freeFrom(other.getFreeFrom())
                    .vat(Objects.toString(section.getVat()))
                    .currency(section.getCurrency())
                    .build();
        }
    }

    @Value
    @Builder(toBuilder = true)
    public static class PickupDeliveryInfo {
        @JsonProperty("address")
        String address;
        @JsonProperty("work_time")
        String workTime;
        @JsonProperty("price")
        BigDecimal price;
        @JsonProperty("free_from")
        BigDecimal freeFrom;
        @JsonProperty("min_delivery_period")
        int minDeliveryPeriod;
        @JsonProperty("max_delivery_period")
        int maxDeliveryPeriod;
        @JsonProperty("vat")
        String vat;
        @Deprecated(forRemoval = true)
        @JsonProperty("currency")
        CurrencyEnum currency;

        public static PickupDeliveryInfo create(DeliverySection section, PickupDeliverySettings other) {
            return PickupDeliveryInfo.builder()
                    .address(other.getAddress())
                    .workTime(other.getWorkTime())
                    .price(other.getPrice())
                    .freeFrom(other.getFreeFrom())
                    .maxDeliveryPeriod(other.getMaxDeliveryPeriod())
                    .minDeliveryPeriod(other.getMinDeliveryPeriod())
                    .vat(Objects.toString(section.getVat()))
                    .currency(section.getCurrency())
                    .build();
        }


    }

    @Value
    @Builder(toBuilder = true)
    public static class CourierDeliveryInfo {
        @JsonProperty("name")
        String name;
        @JsonProperty("min_delivery_period")
        int minDeliveryPeriod;
        @JsonProperty("max_delivery_period")
        int maxDeliveryPeriod;
        @JsonProperty("price")
        BigDecimal price;
        @JsonProperty("free_from")
        BigDecimal freeFrom;
        @JsonProperty("vat")
        String vat;
        @Deprecated(forRemoval = true)
        @JsonProperty("currency")
        CurrencyEnum currency;

        public static CourierDeliveryInfo create(DeliverySection section, CourierDeliverySettings other) {
            return CourierDeliveryInfo.builder()
                    .name(other.getName())
                    .minDeliveryPeriod(other.getMinDeliveryPeriod())
                    .maxDeliveryPeriod(other.getMaxDeliveryPeriod())
                    .price(other.getPrice())
                    .freeFrom(other.getFreeFrom())
                    .vat(Objects.toString(section.getVat()))
                    .currency(section.getCurrency())
                    .build();
        }
    }
}
