package ru.yandex.webmaster3.storage.turbo.service.preview.commerce;

import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.node.ObjectNode;
import lombok.Builder;
import lombok.Value;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.webmaster3.core.turbo.model.TurboHostSettings;
import ru.yandex.webmaster3.core.turbo.model.analytics.AnalyticsSettings;
import ru.yandex.webmaster3.core.turbo.model.app.TurboAppSettings;
import ru.yandex.webmaster3.core.turbo.model.commerce.TurboCommerceSettings;
import ru.yandex.webmaster3.core.turbo.model.feedback.TurboFeedbackButton;
import ru.yandex.webmaster3.storage.turbo.service.preview.TurboHostSettingsData;
import ru.yandex.webmaster3.storage.turbo.service.preview.TurboMenuItemData;

import static ru.yandex.webmaster3.core.turbo.model.feedback.TurboFeedbackButtonType.CALL;
import static ru.yandex.webmaster3.core.turbo.model.feedback.TurboFeedbackButtonType.MAIL;

/**
 * Created by Oleg Bazdyrev on 18/06/2020.
 * Все настройки нового e-commerce в одном месте
 */
@Value
@Builder
public class NewCommerceSettingsData {

    // from turbo app
    @JsonProperty("enabled")
    boolean enabled;
    @JsonProperty("title")
    String title;
    @JsonProperty("description")
    String description;
    @JsonProperty("logo")
    TurboHostSettingsData.Logo logo;
    @JsonProperty("theme")
    String theme;
    // common old settings
    @JsonProperty("analytics")
    List<ObjectNode> analytics;
    @JsonProperty("menu")
    List<TurboMenuItemData> menu;
    @JsonProperty("category_info")
    List<TurboMenuItemData> autoMenu;
    @JsonProperty("feedback_mail")
    String feedbackEmail;
    @JsonProperty("feedback_call")
    String feedbackPhone;
    // from product info
    @JsonProperty("cart_url")
    String cartUrl;
    @JsonProperty("accordion")
    List<ProductInfoSection> productInfoSections;
    @JsonProperty("order")
    OrderInfo orderInfo;
    @JsonProperty("enable_one_click_buy")
    boolean checkoutEmailEnabled;
    @JsonProperty("enable_turbo_cart")
    boolean turboCartEnabled;
    @JsonProperty("enable_turbo_listings")
    boolean turboListingsEnabled;
    @JsonProperty("enable_payment")
    boolean paymentEnabled;
    @JsonProperty("payment")
    PaymentInfo paymentInfo;
    @JsonProperty("shop_info")
    ShopInfo shopInfo;
    @JsonProperty("delivery")
    DeliveryInfo deliveryInfo;
    @JsonProperty("enable_turbo_main_page")
    boolean turboMainPageEnabled;


    public static NewCommerceSettingsData fromInternalSettings(TurboHostSettings settings, TurboAppSettings appSettings) {
        var builder = builder();
        if (appSettings != null) {
            builder.enabled(Boolean.TRUE.equals(appSettings.getEnabled()));
            builder.title(appSettings.getTitle());
            builder.description(appSettings.getDescription());
            builder.theme(appSettings.getTurboColorsString());
            builder.logo(TurboHostSettingsData.Logo.fromTurboLogo(appSettings.getLogoInfo()));
        }
        if (settings != null && settings.getCommerceSettings() != null) {
            // TODO у магазинов будет своя отдельная аналитика?
            builder.analytics(settings.getAnalyticsSettings().stream().map(AnalyticsSettings::toJson).collect(Collectors.toList()));
            // TODO у магазинов будет свое отдельное меню
            List<TurboMenuItemData> menu = TurboMenuItemData.fromMenu(settings.getTopMenuSettings(), false);
            menu.addAll(TurboMenuItemData.fromMenu(settings.getMenuSettings(), false));
            builder.menu(menu);
            builder.autoMenu(TurboMenuItemData.fromMenu(settings.getAutoMenuSettings(), true));
            if (settings.getFeedbackSettings() != null && !CollectionUtils.isEmpty(settings.getFeedbackSettings().getButtons())) {
                var buttons = settings.getFeedbackSettings().getButtons();
                builder.feedbackEmail(buttons.stream().filter(b -> b.getType() == MAIL).map(TurboFeedbackButton::getUrl).findAny().orElse(null));
                builder.feedbackPhone(buttons.stream().filter(b -> b.getType() == CALL).map(TurboFeedbackButton::getUrl).findAny().orElse(null));
            }

            // from product info
            TurboCommerceSettings commerceSettings = settings.getCommerceSettings();
            if (commerceSettings.getInfoSections() != null) {
                builder.productInfoSections(commerceSettings.getInfoSections().stream().map(ProductInfoSection::fromCommerceInfoSection)
                        .collect(Collectors.toList()));
            }
            if (commerceSettings.isCartUrlEnabled()) {
                builder.cartUrl(commerceSettings.getCartUrl());
            }
            builder.cartUrl(commerceSettings.isCartUrlEnabled() ? commerceSettings.getCartUrl() : null);
            Pair<Boolean, PaymentInfo> pair = PaymentInfo.fromPaymentsSettings(commerceSettings.getPaymentsSettings());
            builder.orderInfo(OrderInfo.fromSettings(settings));
            builder.checkoutEmailEnabled(commerceSettings.isCheckoutEmailEnabled());
            builder.turboCartEnabled(commerceSettings.isTurboCartEnabled());
            builder.turboListingsEnabled(commerceSettings.isTurboListingsEnabled());
            builder.paymentEnabled(pair.getLeft());
            builder.paymentInfo(pair.getRight());
            builder.shopInfo(ShopInfo.fromPaymentsSettings(commerceSettings.getPaymentsSettings()));
            builder.deliveryInfo(DeliveryInfo.create(commerceSettings.getDeliverySection()));
            builder.turboMainPageEnabled(Boolean.TRUE.equals(commerceSettings.getAutoMorda()));

        }
        return builder.build();
    }
}
