package ru.yandex.webmaster3.storage.turbo.service.preview.commerce;

import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import lombok.AccessLevel;
import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.experimental.FieldDefaults;
import lombok.experimental.SuperBuilder;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.webmaster3.core.turbo.model.TurboHostSettings;
import ru.yandex.webmaster3.core.turbo.model.commerce.TurboCommerceSettings;
import ru.yandex.webmaster3.core.turbo.model.commerce.delivery.DeliverySection;
import ru.yandex.webmaster3.storage.turbo.service.TurboDomainsStateService;
import ru.yandex.webmaster3.storage.turbo.service.TurboDomainsStateService.TurboDomainState;

/**
 * Created by Oleg Bazdyrev on 19/11/2018.
 */
@Getter
@FieldDefaults(makeFinal = true, level = AccessLevel.PRIVATE)
@SuperBuilder
@AllArgsConstructor(onConstructor_ = @JsonCreator)
public class ProductInfo {

    @JsonProperty("cart_url")
    String cartUrl;
    @JsonProperty("accordion")
    List<ProductInfoSection> productInfoSections;
    @JsonProperty("order")
    OrderInfo orderInfo;
    @JsonProperty("enable_one_click_buy")
    boolean checkoutEmailEnabled; // ака один клик
    @JsonProperty("enable_turbo_cart")
    boolean turboCartEnabled; // турбо-корзина
    @JsonProperty("enable_turbo_listings")
    boolean turboListingsEnabled; // турбо-листинг
    @JsonProperty("enable_payment")
    boolean paymentEnabled;
    @JsonProperty("payment")
    PaymentInfo paymentInfo;
    @JsonProperty("shop_info")
    ShopInfo shopInfo;
    @JsonProperty("delivery")
    DeliveryInfo deliveryInfo;
    @JsonProperty("enable_turbo_main_page")
    boolean turboMainPageEnabled;
    @JsonProperty("market_partner_id")
    Long marketPartnerId;
    @JsonProperty("market_delivery_enabled")
    boolean marketDeliveryEnabled;

    public static ProductInfo fromCommerceSettings(TurboHostSettings settings, TurboDomainState domainState) {
        TurboCommerceSettings commerceSettings = settings.getCommerceSettings();
        if (commerceSettings == null) {
            return null;
        }

        final boolean marketDeliveryEnabled = Optional.ofNullable(commerceSettings.getDeliverySection())
                .map(DeliverySection::getMarketDeliveryEnabled)
                .orElse(false);

        List<ProductInfoSection> accordion = null;
        if (commerceSettings.getInfoSections() != null) {
            accordion = commerceSettings.getInfoSections().stream().map(ProductInfoSection::fromCommerceInfoSection)
                    .collect(Collectors.toList());
        }
        String cartUrl = commerceSettings.isCartUrlEnabled() ? commerceSettings.getCartUrl() : null;
        Pair<Boolean, PaymentInfo> pair = PaymentInfo.fromPaymentsSettings(commerceSettings.getPaymentsSettings());
        ShopInfo shopInfo = ShopInfo.fromPaymentsSettings(commerceSettings.getPaymentsSettings());
        Long marketPartnerId = Optional.ofNullable(domainState)
                .flatMap(st -> st.getMarketFeeds().stream().map(TurboDomainsStateService.TurboMarketFeed::getPartnerId).findAny()).orElse(null);

        var builder = ProductInfo.builder()
                .cartUrl(cartUrl)
                .productInfoSections(accordion)
                .orderInfo(OrderInfo.fromSettings(settings))
                .checkoutEmailEnabled(commerceSettings.isCheckoutEmailEnabled())
                .turboCartEnabled(commerceSettings.isTurboCartEnabled())
                .turboListingsEnabled(commerceSettings.isTurboListingsEnabled())
                .turboMainPageEnabled(Boolean.TRUE.equals(commerceSettings.getAutoMorda()))
                .marketPartnerId(marketPartnerId)
                .marketDeliveryEnabled(marketDeliveryEnabled)
                ;
        // WMC-10124: при отключенной турбо-корзине не должно быть включено ничего лишнего
        boolean turboCartOn = commerceSettings.isCheckoutEmailEnabled() || commerceSettings.isTurboCartEnabled();
        if (turboCartOn) {
            builder
                    .paymentEnabled(pair.getLeft())
                    .paymentInfo(pair.getRight())
                    .shopInfo(shopInfo)
                    .deliveryInfo(DeliveryInfo.create(commerceSettings.getDeliverySection()));
        }
        return builder.build();
    }
}
