package ru.yandex.webmaster3.storage.turbo.service.settings;

import java.net.IDN;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.http.HttpStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.log.EventLogService;
import ru.yandex.webmaster3.core.metrika.counters.CounterBindingStateEnum;
import ru.yandex.webmaster3.core.turbo.TurboConstants;
import ru.yandex.webmaster3.core.turbo.model.TurboAutoRelatedSettings;
import ru.yandex.webmaster3.core.turbo.model.TurboHostHeaderType;
import ru.yandex.webmaster3.core.turbo.model.TurboHostSettings;
import ru.yandex.webmaster3.core.turbo.model.TurboHostSettings.TurboHostSettingsBuilder;
import ru.yandex.webmaster3.core.turbo.model.TurboLogo;
import ru.yandex.webmaster3.core.turbo.model.analytics.AnalyticsSettings;
import ru.yandex.webmaster3.core.turbo.model.app.TurboAppSettings;
import ru.yandex.webmaster3.core.turbo.model.autoparser.AutoparserToggleState;
import ru.yandex.webmaster3.core.turbo.model.commerce.TurboColorScheme;
import ru.yandex.webmaster3.core.turbo.model.commerce.TurboCommerceSettings;
import ru.yandex.webmaster3.core.turbo.model.commerce.TurboCommerceSettings.TurboCommerceSettingsBuilder;
import ru.yandex.webmaster3.core.turbo.model.commerce.TurboNewCommerceSettings;
import ru.yandex.webmaster3.core.turbo.model.commerce.TurboPlusSettings;
import ru.yandex.webmaster3.core.turbo.model.desktop.TurboDesktopSettings;
import ru.yandex.webmaster3.core.turbo.model.desktop.TurboDesktopSettings.TurboDesktopSettingsBuilder;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedSettings;
import ru.yandex.webmaster3.core.turbo.model.feedback.TurboFeedbackButton;
import ru.yandex.webmaster3.core.turbo.model.feedback.TurboFeedbackButtonType;
import ru.yandex.webmaster3.core.turbo.model.feedback.TurboFeedbackSettings;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.storage.avatars.AvatarImageStoreService;
import ru.yandex.webmaster3.storage.avatars.AvatarPicture;
import ru.yandex.webmaster3.storage.avatars.UploadPictureResult;
import ru.yandex.webmaster3.storage.events.data.WMCEventContent;
import ru.yandex.webmaster3.storage.events.data.events.RetranslateToUsersEvent;
import ru.yandex.webmaster3.storage.events.data.events.TurboSettingsChangeEvent;
import ru.yandex.webmaster3.storage.events.data.events.UserDomainMessageEvent;
import ru.yandex.webmaster3.storage.events.service.WMCEventsService;
import ru.yandex.webmaster3.storage.favicon.FaviconInfo;
import ru.yandex.webmaster3.storage.favicon.HostFaviconsCHDao;
import ru.yandex.webmaster3.storage.metrika.dao.MetrikaCounterBindingStateYDao;
import ru.yandex.webmaster3.storage.settings.history.SettingsHistoryService;
import ru.yandex.webmaster3.storage.turbo.dao.TurboDesktopSettingsYDao;
import ru.yandex.webmaster3.storage.turbo.dao.TurboDomainSettingsYDao;
import ru.yandex.webmaster3.storage.turbo.dao.app.TurboAppSettingsYDao;
import ru.yandex.webmaster3.storage.turbo.dao.commerce.TurboPlusSettingsYDao;
import ru.yandex.webmaster3.storage.turbo.dao.logo.DefaultTurboLogoYDao;
import ru.yandex.webmaster3.storage.turbo.logo.TurboLogoData;
import ru.yandex.webmaster3.storage.turbo.logo.TurboLogoYDao;
import ru.yandex.webmaster3.storage.turbo.service.TurboFeedsService;
import ru.yandex.webmaster3.storage.turbo.service.autoparser.TurboAutoparserInfoService;
import ru.yandex.webmaster3.storage.turbo.service.validation.TurboParserException;
import ru.yandex.webmaster3.storage.turbo.service.validation.TurboParserService;
import ru.yandex.webmaster3.storage.user.message.content.MessageContent;
import ru.yandex.webmaster3.storage.user.notification.NotificationType;

import static ru.yandex.webmaster3.core.turbo.TurboConstants.MIN_FAVICON_SIZE_FOR_LOGO;
import static ru.yandex.webmaster3.core.turbo.model.feedback.TurboFeedbackButtonType.CALL;
import static ru.yandex.webmaster3.core.turbo.model.feedback.TurboFeedbackButtonType.MAIL;

/**
 * Created by Oleg Bazdyrev on 06/02/2019.
 */
@Slf4j
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class TurboSettingsService {

    private final AvatarImageStoreService avatarImageStoreService;
    private final DefaultTurboLogoYDao defaultTurboLogoYDao;
    private final MetrikaCounterBindingStateYDao metrikaCounterBindingStateYDao;
    private final HostFaviconsCHDao mdbHostFaviconsCHDao;
    private final TurboAppSettingsYDao turboAppSettingsYDao;
    private final TurboAutoparserInfoService turboAutoparserInfoService;
    private final TurboDesktopSettingsYDao turboDesktopSettingsYDao;
    private final TurboDomainSettingsYDao turboDomainSettingsYDao;
    private final TurboFeedsService turboFeedsService;
    private final TurboLogoYDao turboLogoYDao;
    private final TurboParserService turboParserService;
    private final TurboPlusSettingsYDao turboPlusSettingsYDao;
    private final WMCEventsService wmcEventsService;
    private final SettingsHistoryService settingsHistoryService;
    private final EventLogService eventLogService;

    public void updateSettings(WebmasterHostId hostId, TurboHostSettings settings)  {
        updateSettings(WwwUtil.cutWWWAndM(hostId.getPunycodeHostname()), settings);
    }

    public void updateSettings(String domain, TurboHostSettings settings, Long userId)  {
        updateSettings(domain, settings);
        settingsHistoryService.updateSettingsTurbo(userId, domain, settings);
    }

    public void updateSettings(String domain, TurboHostSettings settings)  {
        turboDomainSettingsYDao.updateSettings(domain, settings);
        notifyAboutSettingsChange(domain, settings, null, null);
    }

    public void updateDesktopSettings(String domain, TurboDesktopSettings settings, Long userId) {
        updateDesktopSettings(domain, settings);
        settingsHistoryService.updateSettingsTurbo(userId, domain, settings);
    }

    public void updateDesktopSettings(String domain, TurboDesktopSettings settings) {
        turboDesktopSettingsYDao.updateSettings(domain, settings);
        notifyAboutSettingsChange(domain, null, null, null);
    }

    @NotNull
    public TurboHostSettings getSettings(WebmasterHostId hostId)  {
        return getSettings(WwwUtil.cutWWWAndM(hostId.getPunycodeHostname()));
    }

    @NotNull
    public TurboHostSettings getSettings(String domain)  {
        TurboHostSettings result = turboDomainSettingsYDao.getSettings(domain);
        if (result == null) {
            TurboHostSettingsBuilder builder = new TurboHostSettingsBuilder();
            builder.setAutoRelated(true);                           // включаем автоленту по умолчанию всем
            builder.setAutoRelatedSettings(new TurboAutoRelatedSettings(TurboAutoRelatedSettings.TurboAutoRelatedType.CARDS));    // включаем новую автоленту по умолчанию всем
            builder.setAnalyticsSettings(getDefaultAnalyticsSettings(domain));

            result = builder.build();
        }
        return result;
    }

    @NotNull
    public TurboDesktopSettings getDesktopSettings(String domain) {
        TurboDesktopSettings result = turboDesktopSettingsYDao.getSettings(domain);
        if (result == null) {
            TurboDesktopSettingsBuilder builder = new TurboDesktopSettingsBuilder();
            result = builder.build();
        }
        return result;
    }

    @NotNull
    public TurboNewCommerceSettings getNewCommerceSettings(String domain) {
        TurboHostSettings domainSettings = turboDomainSettingsYDao.getSettings(domain);
        return getNewCommerceSettings(domain, domainSettings);
    }

    @NotNull
    public TurboNewCommerceSettings getNewCommerceSettings(String domain, TurboHostSettings domainSettings) {
        TurboAppSettings appSettings = turboAppSettingsYDao.getSettings(domain);
        TurboCommerceSettings commerceSettings = domainSettings == null ? null : domainSettings.getCommerceSettings();
        TurboLogo logoInfo = appSettings == null ? null : appSettings.getLogoInfo();
        if (logoInfo == null) {
            logoInfo = getDefaultCommerceLogo(domain);
        }
        TurboColorScheme theme = appSettings == null ? null : TurboColorScheme.fromString(appSettings.getTheme());

        // склеиваем все воедино
        var builder = TurboNewCommerceSettings.builder().theme(theme).logoInfo(logoInfo);

        if (appSettings != null) {
            builder.enabled(appSettings.getEnabled())
                    .title(appSettings.getTitle())
                    .description(appSettings.getDescription());
        }
        if (commerceSettings != null) {
            builder.cartUrl(commerceSettings.getCartUrl())
                    .cartUrlType(commerceSettings.getCartUrlType())
                    .cartUrlEnabled(commerceSettings.isCartUrlEnabled())
                    .checkoutEmail(commerceSettings.getCheckoutEmail())
                    .checkoutEmailEnabled(commerceSettings.isCheckoutEmailEnabled())
                    .infoSections(commerceSettings.getInfoSections())
                    .parsedAccordion(commerceSettings.getParsedAccordion())
                    .turboCartEnabled(commerceSettings.isTurboCartEnabled())
                    .turboListingsEnabled(commerceSettings.isTurboListingsEnabled())
                    .paymentsSettings(commerceSettings.getPaymentsSettings())
                    .deliverySection(commerceSettings.getDeliverySection())
                    .autoMorda(commerceSettings.getAutoMorda())
                    .bitrixSettings(commerceSettings.getBitrixSettings())
                    .marketRatingShow(Objects.requireNonNullElse(commerceSettings.getMarketRatingShow(), true));
        }
        if (domainSettings != null && domainSettings.getFeedbackSettings() != null && domainSettings.getFeedbackSettings().getButtons() != null) {
            domainSettings.getFeedbackSettings().getButtons().stream().filter(button -> button.getType() == MAIL).findAny()
                    .ifPresent(button -> builder.feedbackEmail(button.getUrl()));
            domainSettings.getFeedbackSettings().getButtons().stream().filter(button -> button.getType() == CALL).findAny()
                    .ifPresent(button -> builder.feedbackPhone(button.getUrl()));
        }
        return builder.build();
    }

    public TurboLogo getDefaultCommerceLogo(String domain) {
        // сперва пытаемся найти фавиконку нужного размера (96+ пикселей)
        // если есть только мелкие фавиконки - кладем ее в неизменном виде на прозрачный фон
        List<WebmasterHostId> hostIds = WwwUtil.domainToHostIds(domain);
        Map<WebmasterHostId, List<FaviconInfo>> faviconsByHostId = mdbHostFaviconsCHDao.getFavicons(hostIds);
        for (WebmasterHostId hostId : hostIds) {
            // даем предпочтение фавиконкам на поиске с максимальным размером
            List<FaviconInfo> favicons = faviconsByHostId.get(hostId);
            favicons.sort(FaviconInfo.BY_SIZE.reversed());
            for (FaviconInfo favicon : favicons) {
                if (favicon.getHttpCode() != HttpStatus.SC_OK && favicon.getHttpCode() != 0) {
                    continue;
                }
                if (favicon.getWidth() >= MIN_FAVICON_SIZE_FOR_LOGO && favicon.getHeight() >= MIN_FAVICON_SIZE_FOR_LOGO) {
                    TurboLogo result = uploadAndSaveLogo(favicon.getUrl(), domain);
                    if (result != null) {
                        turboAppSettingsYDao.updateLogo(domain, result);
                        return result;
                    }
                }
            }
        }
        // если фавиконок нет вообще - используем синюю букву
        return defaultTurboLogoYDao.getLogo(IDN.toUnicode(domain).substring(0, 1), "blue");
    }

    private TurboLogo uploadAndSaveLogo(String url, String domain) {
        try {
            UploadPictureResult result = avatarImageStoreService.uploadPicture(url, (Integer) null);
            if (!CollectionUtils.isEmpty(result.getPictures())) {
                AvatarPicture logo = result.getPictures().get(0);
                String logoId = logo.getFilename();
                long groupId = logo.getGroupId();
                log.debug("Generated logo id: {}, group id: {}", logoId, groupId);
                String publicUrlMds = avatarImageStoreService.getPictureUrl(logo);
                log.debug("Mds logo url: {}", publicUrlMds);

                TurboLogoData logoToSave = new TurboLogoData(IdUtils.urlToHostId(domain), logoId, groupId,
                        publicUrlMds, DateTime.now(), false,
                        result.getWidth(), result.getHeight());
                turboLogoYDao.add(logoToSave);
                EnumMap<TurboHostHeaderType, String> logoUrls = new EnumMap<>(TurboHostHeaderType.class);
                for (TurboHostHeaderType type : TurboHostHeaderType.values()) {
                    if (type.getLogoSize() != null) {
                        logoUrls.put(type, avatarImageStoreService.getPictureUrl(logo, result.isSvg() ? "svg" :
                                type.getLogoSize()));
                    }
                }
                return TurboLogo.builder()
                        .logoId(logoId + TurboLogoData.FRONT_LOGO_ID_SEPARATOR + groupId)
                        .logoUrls(logoUrls)
                        .height(result.getHeight())
                        .width(result.getWidth())
                        .svg(result.isSvg())
                        .build();
            }
        } catch (Exception e) {
            log.error("Error when uploading favicon as turbo logo: ", e);
        }
        return null;
    }

    public void splitSettings(@NotNull TurboNewCommerceSettings settings,
                              @NotNull TurboAppSettings.TurboAppSettingsBuilder appSettingsBuilder,
                              @NotNull TurboHostSettingsBuilder hostSettingsBuilder) {
        // раскидываем обратно настройки
        appSettingsBuilder
                .enabled(settings.getEnabled())
                .title(settings.getTitle())
                .description(settings.getDescription())
                .theme(JsonMapping.writeValueAsString(settings.getTheme()))
                .logoInfo(settings.getLogoInfo());

        // настройки е-коммерс
        TurboCommerceSettingsBuilder commerceSettingsBuilder =
                new TurboCommerceSettingsBuilder(hostSettingsBuilder.getCommerceSettings());
        // обновляем настройки
        commerceSettingsBuilder
                .setCartUrlEnabled(settings.isCartUrlEnabled())
                .setCartUrl(settings.getCartUrl())
                .setCartUrlType(settings.getCartUrlType())
                .setCheckoutEmail(settings.getCheckoutEmail())
                .setCheckoutEmailEnabled(Boolean.TRUE.equals(settings.getCheckoutEmailEnabled()))
                .setInfoSections(settings.getInfoSections())
                .setParsedAccordion(settings.getParsedAccordion())
                .setTurboCartEnabled(Boolean.TRUE.equals(settings.getTurboCartEnabled()))
                .setTurboListingsEnabled(Boolean.TRUE.equals(settings.getTurboListingsEnabled()))
                .setPaymentsSettings(settings.getPaymentsSettings())
                .setDeliverySection(settings.getDeliverySection())
                .setAutoMorda(settings.getAutoMorda())
                .setBitrixSettings(settings.getBitrixSettings())
                .setMarketingRatingShow(settings.getMarketRatingShow());
        hostSettingsBuilder.setCommerceSettings(commerceSettingsBuilder.build());
        // copy email/phone
        if (hostSettingsBuilder.getFeedbackSettings() == null || hostSettingsBuilder.getFeedbackSettings().getButtons() == null) {
            hostSettingsBuilder.setFeedbackSettings(new TurboFeedbackSettings(TurboFeedbackSettings.Stick.RIGHT,
                    new ArrayList<>()));
        }
        List<TurboFeedbackButton> buttons = hostSettingsBuilder.getFeedbackSettings().getButtons();
        replaceFeedbackButton(CALL, settings.getFeedbackPhone(), buttons);
        replaceFeedbackButton(MAIL, settings.getFeedbackEmail(), buttons);
    }

    public void updateSettings(@NotNull String domain, @NotNull TurboNewCommerceSettings settings) {
        TurboAppSettings.TurboAppSettingsBuilder appSettingsBuilder = TurboAppSettings.builder();
        TurboHostSettingsBuilder hostSettingsBuilder = new TurboHostSettingsBuilder(turboDomainSettingsYDao.getSettings(domain));
        splitSettings(settings, appSettingsBuilder, hostSettingsBuilder);

        TurboAppSettings appSettings = appSettingsBuilder.build();
        turboAppSettingsYDao.updateSettings(domain, appSettings);

        TurboHostSettings hostSettings = hostSettingsBuilder.build();
        turboDomainSettingsYDao.updateSettings(domain, hostSettings);
        notifyAboutSettingsChange(domain, hostSettings, null, null, null, appSettings, null);
    }

    public void updateSettings(@NotNull String domain, @NotNull TurboAppSettings appSettings,
                               @NotNull TurboHostSettings hostSettings) {
        turboAppSettingsYDao.updateSettings(domain, appSettings);
        turboDomainSettingsYDao.updateSettings(domain, hostSettings);
        notifyAboutSettingsChange(domain, hostSettings, null, null, null, appSettings, null);
    }

    private void replaceFeedbackButton(TurboFeedbackButtonType type, String value, List<TurboFeedbackButton> buttons) {
        // replace button
        TurboFeedbackButton phoneButton = TurboFeedbackButton.builder().type(type).url(value).build();
        boolean found = false;
        for (int i = 0; i < buttons.size(); i++) {
            TurboFeedbackButton button = buttons.get(i);
            if (button.getType() == type) {
                if (!Strings.isNullOrEmpty(value)) {
                    buttons.set(i, phoneButton);
                } else {
                    buttons.remove(i);
                }
                found = true;
                break;
            }
        }
        if (!found && !Strings.isNullOrEmpty(value)) {
            buttons.add(phoneButton);
        }
    }

    private List<AnalyticsSettings> getDefaultAnalyticsSettings(String domain) {
        return metrikaCounterBindingStateYDao.getAllForDomainAsMap(domain).values().stream()
                .filter(cb -> cb.getCounterBindingState() == CounterBindingStateEnum.APPROVED)
                .map(cb -> new AnalyticsSettings.MetrikaCounterSettings(String.valueOf(cb.getCounterId()), false))
                .map(source -> {
                    try {
                        return turboParserService.parseAnalyticsSettings(source);
                    } catch (TurboParserException e) {
                        return null; // impossible
                    }
                })
                .filter(Objects::nonNull).limit(TurboConstants.MAX_ANALYTICS_SETINGS).collect(Collectors.toList());
    }

    public void notifyAboutSettingsChange(@NotNull String domain,
                                          @Nullable TurboHostSettings turboHostSettings,
                                          @Nullable AutoparserToggleState autoparserToggleState,
                                          @Nullable List<TurboFeedSettings> feedSettingsList) {
        notifyAboutSettingsChange(domain, turboHostSettings, null, autoparserToggleState, feedSettingsList);
    }

    public void notifyAboutSettingsChange(@NotNull String domain,
                                          @Nullable TurboHostSettings turboHostSettings,
                                          @Nullable TurboDesktopSettings turboDesktopSettings,
                                          @Nullable AutoparserToggleState autoparserToggleState,
                                          @Nullable List<TurboFeedSettings> feedSettingsList) {
        notifyAboutSettingsChange(domain, turboHostSettings, turboDesktopSettings, autoparserToggleState,
                feedSettingsList, null, null);
    }

    public void notifyAboutSettingsChange(@NotNull String domain,
                                          @Nullable TurboHostSettings turboHostSettings,
                                          @Nullable TurboDesktopSettings turboDesktopSettings,
                                          @Nullable AutoparserToggleState autoparserToggleState,
                                          @Nullable List<TurboFeedSettings> feedSettingsList,
                                          @Nullable TurboAppSettings appSettings,
                                          @Nullable TurboPlusSettings turboPlusSettings) {
        if (turboHostSettings == null) {
            turboHostSettings = getSettings(domain);
        }
        if (turboDesktopSettings == null) {
            turboDesktopSettings = getDesktopSettings(domain);
        }
        if (appSettings == null) {
            appSettings = turboAppSettingsYDao.getSettings(domain);
        }
        if (turboPlusSettings == null) {
            turboPlusSettings = turboPlusSettingsYDao.getSettings(domain);
        }
        if (autoparserToggleState == null) {
            autoparserToggleState = turboAutoparserInfoService.getAutoparseCheckBoxState(domain);
        }
        if (feedSettingsList == null) {
            feedSettingsList = turboFeedsService.getFeeds(domain);
        }
        final TurboSettingsChangeEvent turboSettingsChangeEvent = new TurboSettingsChangeEvent(domain,
                turboHostSettings,
                turboDesktopSettings, appSettings, turboPlusSettings, autoparserToggleState, feedSettingsList);
        eventLogService.sendEvent(turboSettingsChangeEvent.convert());
    }

    public void sendNotificationsAboutImportantChanges(
            String domain, long excludeUserId, TurboHostSettingsBuilder settingsBuilder,
            TurboDesktopSettingsBuilder desktopSettingsBuilder) {
        WebmasterHostId hostId = IdUtils.urlToHostId(domain);
        List<WMCEventContent> events = new ArrayList<>();
        var event = UserDomainMessageEvent.create(hostId, MessageContent.TurboAdvSettingsChanged.create(hostId,
                settingsBuilder.getChangedAdvertisingPlacements(),
                desktopSettingsBuilder.getChangedAdvertisingPlacements()), NotificationType.TURBO_ADV_SETTINGS_CHANGE
                , false);
        events.add(new RetranslateToUsersEvent<>(event, Collections.singletonList(excludeUserId)));
        wmcEventsService.addEventContents(events);
    }

    public void sendNotificationsAboutImportantCommerceChanges(String domain, long excludeUserId,
                                                               TurboHostSettingsBuilder settingsBuilder) {
        WebmasterHostId hostId = IdUtils.urlToHostId(domain);
        List<WMCEventContent> events = new ArrayList<>();
        var event = UserDomainMessageEvent.create(
                hostId, new MessageContent.TurboCommerceSettingsChanged(hostId),
                NotificationType.TURBO_COMMERCE_SETTINGS_CHANGE, false);
        events.add(new RetranslateToUsersEvent<>(event, Collections.singletonList(excludeUserId)));
        wmcEventsService.addEventContents(events);
    }
}
