package ru.yandex.webmaster3.storage.turbo.service.validation;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.node.TextNode;
import com.fasterxml.jackson.module.paramnames.ParameterNamesModule;
import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import lombok.RequiredArgsConstructor;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpStatus;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;
import org.jetbrains.annotations.NotNull;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.w3c.dom.Document;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.blackbox.UserWithLogin;
import ru.yandex.webmaster3.core.data.HttpCodeInfo;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.data.WebmasterHostId.Schema;
import ru.yandex.webmaster3.core.data.WebmasterUser;
import ru.yandex.webmaster3.core.host.service.HostOwnerService;
import ru.yandex.webmaster3.core.http.HttpConstants;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse.SitaErrorResponse;
import ru.yandex.webmaster3.core.metrics.externals.AbstractExternalAPIService;
import ru.yandex.webmaster3.core.metrics.externals.ExternalDependencyMethod;
import ru.yandex.webmaster3.core.payments.ServiceMerchantInfo;
import ru.yandex.webmaster3.core.turbo.TurboConstants;
import ru.yandex.webmaster3.core.turbo.adv.AdvertisingIntegrationService;
import ru.yandex.webmaster3.core.turbo.adv.model.response.ExtendedAttributeInfoResponse;
import ru.yandex.webmaster3.core.turbo.model.TurboHostSettingsBlock;
import ru.yandex.webmaster3.core.turbo.model.TurboUserAgreement;
import ru.yandex.webmaster3.core.turbo.model.advertising.AdvertisingNetworkType;
import ru.yandex.webmaster3.core.turbo.model.advertising.AdvertisingPlacement;
import ru.yandex.webmaster3.core.turbo.model.advertising.AdvertisingSettings;
import ru.yandex.webmaster3.core.turbo.model.analytics.AnalyticsSettings;
import ru.yandex.webmaster3.core.turbo.model.commerce.TurboBitrixSettings;
import ru.yandex.webmaster3.core.turbo.model.commerce.TurboCommerceInfoSection;
import ru.yandex.webmaster3.core.turbo.model.commerce.TurboCommerceSettings;
import ru.yandex.webmaster3.core.turbo.model.commerce.TurboCommerceSettings.TurboCommerceSettingsBuilder;
import ru.yandex.webmaster3.core.turbo.model.commerce.TurboPaymentsSettings;
import ru.yandex.webmaster3.core.turbo.model.commerce.delivery.DeliverySection;
import ru.yandex.webmaster3.core.turbo.model.feedback.TurboFeedbackButton;
import ru.yandex.webmaster3.core.turbo.model.feedback.TurboFeedbackSettings;
import ru.yandex.webmaster3.core.turbo.model.menu.TurboMenuItem;
import ru.yandex.webmaster3.core.turbo.model.search.TurboSearchSettings;
import ru.yandex.webmaster3.core.turbo.xml.TurboXMLReader;
import ru.yandex.webmaster3.core.util.EmailValidator;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.JavaMethodWitness;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.core.zora.ZoraForValidatorsService;
import ru.yandex.webmaster3.storage.abt.AbtService;
import ru.yandex.webmaster3.storage.abt.model.Experiment;
import ru.yandex.webmaster3.storage.payments.PaymentsService;
import ru.yandex.webmaster3.storage.payments.ServiceMerchantCacheYDao;
import ru.yandex.webmaster3.storage.turbo.service.commerce.TurboCommerceService;
import ru.yandex.webmaster3.storage.turbo.service.css.TurboCssValidatorResponse;
import ru.yandex.webmaster3.storage.user.UserPersonalInfo;
import ru.yandex.webmaster3.storage.user.UserTakeoutDataProvider;
import ru.yandex.webmaster3.storage.user.service.UserPersonalInfoService;
import ru.yandex.wmtools.common.util.uri.WebmasterUriUtils;

import static ru.yandex.webmaster3.core.turbo.TurboConstants.PLACEHOLDER_SEARCH_URL_TEXT;
import static ru.yandex.webmaster3.core.turbo.model.advertising.AdvertisingNetworkType.ADFOX;
import static ru.yandex.webmaster3.core.turbo.model.advertising.AdvertisingNetworkType.ADFOX_INPAGE;
import static ru.yandex.webmaster3.core.turbo.model.advertising.AdvertisingNetworkType.ADFOX_INSTREAM;
import static ru.yandex.webmaster3.core.turbo.model.advertising.AdvertisingNetworkType.ADFOX_INTERSCROLLER;
import static ru.yandex.webmaster3.storage.turbo.service.validation.PartnerBlockConstants.POSSIBLE_FORMATS_BY_PLACEMENT;
import static ru.yandex.webmaster3.storage.turbo.service.validation.PartnerBlockConstants.POSSIBLE_FORMATS_BY_PLACEMENT_MOBILE;
import static ru.yandex.webmaster3.storage.turbo.service.validation.PartnerBlockConstants.POSSIBLE_MEDIA_SIZES_BY_PLACEMENT;
import static ru.yandex.webmaster3.storage.turbo.service.validation.PartnerBlockConstants.POSSIBLE_MEDIA_SIZES_BY_PLACEMENT_MOBILE;
import static ru.yandex.webmaster3.storage.turbo.service.validation.PartnerBlockConstants.SITE_VERSION_DESKTOP;
import static ru.yandex.webmaster3.storage.turbo.service.validation.PartnerBlockConstants.SITE_VERSION_TURBO_DESKTOP;
import static ru.yandex.webmaster3.storage.turbo.service.validation.TurboParserException.ErrorCode.ADFOX_INPAGE_INVALID_CODE;
import static ru.yandex.webmaster3.storage.turbo.service.validation.TurboParserException.ErrorCode.ADFOX_INSTREAM_INVALID_CODE;
import static ru.yandex.webmaster3.storage.turbo.service.validation.TurboParserException.ErrorCode.ADFOX_INTERSCROLLER_INVALID_CODE;
import static ru.yandex.webmaster3.storage.turbo.service.validation.TurboParserException.ErrorCode.ADFOX_INVALID_CODE;
import static ru.yandex.webmaster3.storage.turbo.service.validation.TurboParserException.ErrorCode.CUSTOM_TRACKER_NOT_URL;
import static ru.yandex.webmaster3.storage.turbo.service.validation.TurboParserException.ErrorCode.DUPLICATE_COMMERCE_SECTION;
import static ru.yandex.webmaster3.storage.turbo.service.validation.TurboParserException.ErrorCode.EMPTY_TURBOCART_EMAIL;
import static ru.yandex.webmaster3.storage.turbo.service.validation.TurboParserException.ErrorCode.INVALID_CART_URL;
import static ru.yandex.webmaster3.storage.turbo.service.validation.TurboParserException.ErrorCode.INVALID_CHECKOUT_EMAIL;
import static ru.yandex.webmaster3.storage.turbo.service.validation.TurboParserException.ErrorCode.INVALID_FEEDBACK_AGREEMENT;
import static ru.yandex.webmaster3.storage.turbo.service.validation.TurboParserException.ErrorCode.INVALID_FEEDBACK_AGREEMENT_URL;
import static ru.yandex.webmaster3.storage.turbo.service.validation.TurboParserException.ErrorCode.INVALID_FEEDBACK_EMAIL;
import static ru.yandex.webmaster3.storage.turbo.service.validation.TurboParserException.ErrorCode.INVALID_FEEDBACK_URL;
import static ru.yandex.webmaster3.storage.turbo.service.validation.TurboParserException.ErrorCode.INVALID_METRIKA_ID;
import static ru.yandex.webmaster3.storage.turbo.service.validation.TurboParserException.ErrorCode.INVALID_MIN_ORDER_VALUE;
import static ru.yandex.webmaster3.storage.turbo.service.validation.TurboParserException.ErrorCode.INVALID_SEARCH_URL;
import static ru.yandex.webmaster3.storage.turbo.service.validation.TurboParserException.ErrorCode.MENU_EMPTY_LABEL;
import static ru.yandex.webmaster3.storage.turbo.service.validation.TurboParserException.ErrorCode.MENU_INVALID_URL;
import static ru.yandex.webmaster3.storage.turbo.service.validation.TurboParserException.ErrorCode.PAYMENTS_TOKEN_NOT_FOUND;
import static ru.yandex.webmaster3.storage.turbo.service.validation.TurboParserException.ErrorCode.YANDEX_AD_NON_TURBO;
import static ru.yandex.webmaster3.storage.turbo.service.validation.TurboParserException.ErrorCode.YANDEX_AD_NOT_EXISTS;
import static ru.yandex.webmaster3.storage.turbo.service.validation.TurboParserException.ErrorCode.YANDEX_INSTREAM_INVALID_ID;
import static ru.yandex.webmaster3.storage.turbo.service.validation.TurboParserException.ErrorCode.YANDEX_REC_WIDGET_INVALID_ID;

/**
 * Сервис для парсинга всяких турбовых штучек (пока только код AdFox)
 * Created by Oleg Bazdyrev on 05/09/2017.
 */
@Component("turboParserService")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class TurboParserService extends AbstractExternalAPIService implements UserTakeoutDataProvider {
    private static final Logger log = LoggerFactory.getLogger(TurboParserService.class);

    private static final int socketTimeoutMs = 5000;
    private static final int connectTimeoutMs = HttpConstants.DEFAULT_CONNECT_TIMEOUT;
    private static final ObjectMapper OM = new ObjectMapper().registerModule(new ParameterNamesModule());
    private static final Pattern YANDEX_AD_PATTERN = Pattern.compile("(\\w+)-(\\w+)-(\\d+)-(\\d+)");
    private static final Pattern YANDEX_INSTREAM_AD_PATTERN = Pattern.compile("\\d+");
    private static final Pattern YANDEX_METRIKA_COUNTER_PATTERN = Pattern.compile("\\d+");
    private static final Pattern YANDEX_REC_WIDGET = Pattern.compile("C-A-\\d+-\\d+");
    private static final Pattern TWO_DIGIT_SPLIT_BY_X = Pattern.compile("(\\d+)(%?)x(\\d+)");
    private static final String YANDEX_INPAGE_TYPE = "VI";
    private static final Set<String> SUPPORTED_SCHEMES = Collections.unmodifiableSet(new HashSet<>(
            Arrays.asList("callto", "irc", "mailto", "sip", "sips", "skype", "tel", "tg", "whatsapp", "viber",
                    "xmpp", "im", "sms", "aim", "gtalk", "fb-messenger", "facetime", "http", "https")
    ));
    private static Map<AdvertisingNetworkType, TurboParserException.ErrorCode> typeErrorCodeMap = Map.of(
            ADFOX, ADFOX_INVALID_CODE,
            ADFOX_INPAGE, ADFOX_INPAGE_INVALID_CODE,
            ADFOX_INSTREAM, ADFOX_INSTREAM_INVALID_CODE,
            ADFOX_INTERSCROLLER, ADFOX_INTERSCROLLER_INVALID_CODE
    );
    private static final Cache<String, Optional<ExtendedAttributeInfoResponse>> partnerServiceCache =
            CacheBuilder.newBuilder()
                    .expireAfterAccess(10, TimeUnit.MINUTES)
                    .maximumSize(1000L)
                    .build();

    private final AbtService abtService;
    private final HostOwnerService hostOwnerService;
    private final ZoraForValidatorsService zoraForValidatorsService;
    private final PaymentsService paymentsService;
    private final ServiceMerchantCacheYDao serviceMerchantCacheYDao;
    private final UserPersonalInfoService userPersonalInfoService;
    private final TurboCommerceService turboCommerceService;
    private final AdvertisingIntegrationService advertisingIntegrationService;

    @Value("${webmaster3.storage.turbo.adfox.validatorServiceUrl}")
    private URI adfoxValidatorServiceUrl;
    @Value("${webmaster3.storage.turbo.css.validatorServiceUrl}")
    private URI cssValidatorServiceUrl;

    private CloseableHttpClient httpClient;
    // немного XPath для вытаскивания нужных параметров офферов
    private XPathExpression offerIdExpr;
    private XPathExpression offerUrlExpr;
    private XPathExpression offerNameExpr;

    public void init() throws XPathExpressionException {
        RequestConfig requestConfig = RequestConfig.custom()
                .setSocketTimeout(socketTimeoutMs)
                .setConnectTimeout(connectTimeoutMs)
                .build();

        httpClient = HttpClients.custom()
                .setDefaultRequestConfig(requestConfig)
                .setConnectionTimeToLive(30, TimeUnit.SECONDS)
                .build();

        XPath xPath = XPathFactory.newInstance().newXPath();
        offerIdExpr = xPath.compile("/yml_catalog/shop/offers/offer/@id");
        offerUrlExpr = xPath.compile("/yml_catalog/shop/offers/offer/url/text()");
        offerNameExpr = xPath.compile("/yml_catalog/shop/offers/offer/name/text()");
    }

    public AdvertisingSettings parseAdvertisingSettings(
            WebmasterHostId hostId, AdvertisingSettings src, boolean desktop) throws TurboParserException {
        if (src.getType() == null) {
            throw new WebmasterException("Type parameter for advertising not set",
                    new WebmasterErrorResponse.IllegalParameterValueResponse(getClass(), "type", "null"));
        }
        switch (src.getPlacement()) {
            case STICKY:
            case MANUAL_STICKY:
                Preconditions.checkArgument(!desktop, "Invalid placement " + src.getPlacement() + " for platform desktop");
                break;
            case BACKGROUND:
                Preconditions.checkArgument(src.getType() == ADFOX && desktop,
                        "Invalid placement " + src.getPlacement() + " for type " + src.getType());
                break;
        }
        boolean validate = abtService.notInExperiment(hostId, Experiment.TURBO_DISABLE_ADS_VALIDATION);
        switch (src.getType()) {
            case YANDEX:
                return parseYandex(src.getPlacement(), src.getValue(), src.getAlias(), desktop, validate);
            case YANDEX_INSTREAM:
                return parseYandexInstream(src.getPlacement(), src.getValue(), src.getAlias(), validate);
            case YANDEX_REC_WIDGET: {
                //TODO пока не сделана задача: WMC-10388, проверяем что реклама соответствует шаблону C-A-0000-0
                final Matcher matcher = YANDEX_REC_WIDGET.matcher(src.getValue());
                if (validate && !matcher.matches()) {
                    throw new TurboParserException("Invalid pattern for REC_WIDGET adv: " + src.getValue(), YANDEX_REC_WIDGET_INVALID_ID);
                }
                ObjectNode data = OM.createObjectNode();
                data.set("id", new TextNode(src.getValue()));
                return new AdvertisingSettings(AdvertisingNetworkType.YANDEX_REC_WIDGET, AdvertisingPlacement.REC_WIDGET, "", src.getValue(), data);
            }
            case ADFOX:
            case ADFOX_INPAGE:
            case ADFOX_INSTREAM:
            case ADFOX_INTERSCROLLER:
                return parseAdFox(WwwUtil.cutWWWAndM(hostId), src.getPlacement(), src.getValue(), src.getAlias(), src.getType(), desktop, validate);
        }
        throw new IllegalStateException("Unsupported advertising network type - " + src.getType());
    }

    /**
     * Парсит код рекламного блока AdFox и возвращает соответствующую структуру (либо кидает исключение в случае ошибки)
     *
     * @param placement
     * @param adFoxCode
     * @return
     */
    @ExternalDependencyMethod("parse-adfox")
    public AdvertisingSettings parseAdFox(String host,
                                          AdvertisingPlacement placement,
                                          String adFoxCode,
                                          String alias,
                                          AdvertisingNetworkType type,
                                          boolean desktop,
                                          boolean validate) throws TurboParserException {
        return trackQuery(new JavaMethodWitness() {
        }, ALL_ERRORS_INTERNAL, () -> {
            HttpPost post = new HttpPost(new URIBuilder(adfoxValidatorServiceUrl)
                    .addParameter("host", host)
                    .addParameter("desktop", Boolean.toString(desktop))
                    .addParameter("desktopPlacement", placement.getDisplayName())
                    .addParameter("adType", type.getAdType())
                    .toString());
            post.setEntity(new StringEntity(adFoxCode, Charsets.UTF_8));
            try (CloseableHttpResponse response = httpClient.execute(post)) {
                if (response.getStatusLine().getStatusCode() != HttpStatus.SC_OK) {
                    throw new TurboParserException("Parser service returns code: " +
                            response.getStatusLine().getStatusCode(), typeErrorCodeMap.get(type));
                }
                ObjectNode rootNode = (ObjectNode) OM.readTree(response.getEntity().getContent());
                ObjectNode errorNode = (ObjectNode) rootNode.get("error");
                if (errorNode != null) {
                    throw new TurboParserException("Parser service returns message: " +
                            errorNode.get("message").textValue(), typeErrorCodeMap.get(type));
                }
                ObjectNode resultNode = (ObjectNode) rootNode.get("result");
                return new AdvertisingSettings(type, placement, alias, adFoxCode, resultNode);
            } catch (Exception e) {
                log.error("Error parsing AdFox code: ", e);
                throw new TurboParserException("Error parsing AdFox code: " + e.getMessage(), e, typeErrorCodeMap.get(type));
            }
        });
    }

    /**
     * Парсит и проверяет идентификатор РСЯ (на самом деле проверяет лишь на то, что идентификатор не пустой)
     */
    private AdvertisingSettings parseYandex(AdvertisingPlacement placement, String code, String alias, boolean desktop, boolean validate)
            throws TurboParserException {
        if (Strings.isNullOrEmpty(code)) {
            throw new WebmasterException("Value parameter for Yandex.Direct not set",
                    new WebmasterErrorResponse.IllegalParameterValueResponse(getClass(), "value", "null"));
        }

        try {

            ExtendedAttributeInfoResponse blockInfo = partnerServiceCache.get(code, () -> advertisingIntegrationService.getBlockInfo(code)
            ).orElse(null);
            if (validate) {
                validatePartnerBlockInfo(blockInfo, placement, desktop);
            }
            ObjectNode data = OM.createObjectNode();
            data.set("id", new TextNode(code));
            return new AdvertisingSettings(AdvertisingNetworkType.YANDEX, placement, alias, code, data);
        } catch (ExecutionException e) {
            if (e.getCause() instanceof TurboParserException) {
                throw (TurboParserException) e.getCause();
            }
            throw new WebmasterException("Execution exception " + e.getMessage(),
                    new WebmasterErrorResponse.TurboErrorResponse(getClass(), null), e.getCause());
        }
    }

    static void validatePartnerBlockInfo(ExtendedAttributeInfoResponse blockInfo, AdvertisingPlacement placement, boolean desktop) throws TurboParserException {
        if (blockInfo == null || Strings.isNullOrEmpty(blockInfo.getData().getId())) {
            throw new TurboParserException("Block not found", YANDEX_AD_NOT_EXISTS);
        }
        if (desktop) {
            validateDesktopPartnerBlockInfo(blockInfo.getData(), placement);
        } else {
            validateMobilePartnerBlockInfo(blockInfo.getData(), placement);
        }
    }


    static void validateMobilePartnerBlockInfo(ExtendedAttributeInfoResponse.ExtendedAttributeData blockInfo, AdvertisingPlacement placement)
            throws TurboParserException {
        if (!PartnerBlockConstants.SITE_VERSION_TURBO.equals(blockInfo.getAttributes().getSiteVersion())) {
            throw new TurboParserException("Block is non turbo", YANDEX_AD_NON_TURBO);
        }
        // дополнительная валидация для перетяжек WMC-7057
        Set<String> possibleDirectFormats = POSSIBLE_FORMATS_BY_PLACEMENT_MOBILE.get(placement);
        final List<String> desingTempalateNames = blockInfo
                .getAttributes()
                .getDesignTemplates()
                .stream()
                .filter(e -> e.getType().equals("tga"))
                .map(e -> e.getDesignSettings().getName())
                .collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(possibleDirectFormats) && !possibleDirectFormats.containsAll(desingTempalateNames)) {
            throw new TurboParserException("Block direct format is not from list " + possibleDirectFormats,
                    YANDEX_AD_NON_TURBO);
        }
        Set<String> possibleMediaSizes = POSSIBLE_MEDIA_SIZES_BY_PLACEMENT_MOBILE.get(placement);

        final Optional<MediaSize> max = blockInfo.getAttributes().getDspBlocks().stream().map(TurboParserService::splitValue)
                .max(Comparator.naturalOrder());
        if (!CollectionUtils.isEmpty(possibleMediaSizes) && !possibleMediaSizes.contains(max.orElse(new MediaSize("", 0l, 0l)).getVal())) {
            throw new TurboParserException("Media size is not from list " + possibleMediaSizes, YANDEX_AD_NON_TURBO);
        }
    }


    public static MediaSize splitValue(String val) {
        final Matcher matcher = TWO_DIGIT_SPLIT_BY_X.matcher(val);
        if (matcher.matches()) {
            final String group = matcher.group(2);
            final String[] xes = val.split("x");
            if (Strings.isNullOrEmpty(group)) {
                return new MediaSize(val, Long.valueOf(xes[0]), Long.valueOf(xes[1]));
            } else {
                return new MediaSize(val, 0, Long.valueOf(xes[1]));
            }
        }
        return new MediaSize(val, 0L, 0L);
    }

    // WMC-7059
    static void validateDesktopPartnerBlockInfo(ExtendedAttributeInfoResponse.ExtendedAttributeData blockInfo, AdvertisingPlacement placement)
            throws TurboParserException {

        //Preconditions.checkState(placement != AdvertisingPlacement.MANUAL);
        if (!SITE_VERSION_TURBO_DESKTOP.equals(blockInfo.getAttributes().getSiteVersion()) && // TODO temp
                !SITE_VERSION_DESKTOP.equals(blockInfo.getAttributes().getSiteVersion())) {
            throw new TurboParserException("Block is non turbo", YANDEX_AD_NON_TURBO);
        }
        if (placement != AdvertisingPlacement.MANUAL) {
            Set<String> possibleDirectFormats = POSSIBLE_FORMATS_BY_PLACEMENT.get(placement);
            final List<String> designTemplateNames = blockInfo
                    .getAttributes()
                    .getDesignTemplates()
                    .stream()
                    .filter(e -> e.getType().equals("tga"))
                    .map(e -> e.getDesignSettings().getName())
                    .collect(Collectors.toList());
            if (!possibleDirectFormats.containsAll(designTemplateNames)) {
                throw new TurboParserException("Block direct format is not from list " + possibleDirectFormats,
                        YANDEX_AD_NON_TURBO);
            }
            Set<String> possibleMediaSizes = POSSIBLE_MEDIA_SIZES_BY_PLACEMENT.get(placement);
            final Optional<MediaSize> max = blockInfo.getAttributes().getDspBlocks().stream().map(TurboParserService::splitValue)
                    .max(Comparator.naturalOrder());

            if (!possibleMediaSizes.contains(max.orElse(new MediaSize("", 0l,0l)).getVal())) {
                throw new TurboParserException("Media size is not from list " + possibleMediaSizes, YANDEX_AD_NON_TURBO);
            }
        }
    }

    /**
     * WMC-7047
     */
    private AdvertisingSettings parseYandexInstream(AdvertisingPlacement placement, String code, String alias, boolean validate)
            throws TurboParserException {
        if (Strings.isNullOrEmpty(code)) {
            throw new WebmasterException("Value parameter for InStream ad not set",
                    new WebmasterErrorResponse.IllegalParameterValueResponse(getClass(), "value", "null"));
        }

        Matcher matcher = YANDEX_INSTREAM_AD_PATTERN.matcher(code);
        if (validate && !matcher.matches()) {
            throw new TurboParserException("Invalid pattern for InStream ad: " + code, YANDEX_INSTREAM_INVALID_ID);
        }
        ObjectNode data = OM.createObjectNode();
        data.set("id", new TextNode(code));
        return new AdvertisingSettings(AdvertisingNetworkType.YANDEX_INSTREAM, placement, alias, code, data);
    }

    public AnalyticsSettings parseAnalyticsSettings(AnalyticsSettings source) throws TurboParserException {
        if (source.getType() == null) {
            throw new WebmasterException("Type parameter for analytics not set",
                    new WebmasterErrorResponse.IllegalParameterValueResponse(getClass(), "type", "null"));
        }
        ObjectNode data = OM.createObjectNode();
        String value = source.getValue();
        switch (source.getType()) {

            case GOOGLE:

                final AnalyticsSettings.GoogleAnalyticsSettings googleSource = (AnalyticsSettings.GoogleAnalyticsSettings) source;
                //нужно по бизнес логике - если есть domain, то нужна strategy, если же нету то не должно быть и стратегии
                if (Objects.nonNull(googleSource.getDomains()) != googleSource.getStrategy().isDomainsRequired()) {
                    String parameterName;
                    String parameterValue;
                    if (googleSource.getStrategy().isDomainsRequired()) {
                        parameterName = "domains";
                        parameterValue = "";
                    } else {
                        parameterName = "strategy";
                        parameterValue = "none";
                    }

                    throw new WebmasterException("Strategy and domains both required", new WebmasterErrorResponse.IllegalParameterValueResponse(
                            getClass(), parameterName, parameterValue
                    ));
                }
                final String dimension = googleSource.getDimension();
                if (dimension != null && !dimension.matches("[1-9]\\d{0,8}")) {
                    throw new WebmasterException("Dimension must be natural number < 1_000_000_000", new WebmasterErrorResponse.IllegalParameterValueResponse(
                            getClass(), "dimension", dimension.toString()
                    ));
                }

                if (Strings.isNullOrEmpty(value)) {
                    throw new WebmasterException("Value parameter for analytics not set: " + source.getType(),
                            new WebmasterErrorResponse.IllegalParameterValueResponse(getClass(), "value", "null"));
                }
                data.set("id", new TextNode(value));
                break;
            case YANDEX:
                if (!YANDEX_METRIKA_COUNTER_PATTERN.matcher(value).matches()) {
                    throw new TurboParserException("Yandex Metrika counter ID must be integer number", INVALID_METRIKA_ID);
                }
                data.set("id", new TextNode(value));
                break;
            case MAILRU:
            case RAMBLER:
            case MEDIASCOPE:
                if (Strings.isNullOrEmpty(value)) {
                    throw new WebmasterException("Value parameter for analytics not set: " + source.getType(),
                            new WebmasterErrorResponse.IllegalParameterValueResponse(getClass(), "value", "null"));
                }
                data.set("id", new TextNode(value));
                break;
            case LIVEINTERNET:
                if (!Strings.isNullOrEmpty(value)) {
                    data.set("params", new TextNode(value));
                }
                break;
            case CUSTOM:
                if (Strings.isNullOrEmpty(value)) {
                    throw new WebmasterException("Value parameter for analytics not set: " + source.getType(),
                            new WebmasterErrorResponse.IllegalParameterValueResponse(getClass(), "value", "null"));
                }
                // проверим, что перед нами корректный URL
                try {
                    WebmasterUriUtils.toOldUri(value);
                } catch (Exception e) {
                    throw new TurboParserException("Custom counter value is not valid URL", CUSTOM_TRACKER_NOT_URL);
                }
                data.set("url", new TextNode(value));
                break;
            default:
                throw new IllegalStateException("Unsupported analytics system type");
        }
        return source;
    }

    @ExternalDependencyMethod("validate-css")
    public TurboCssValidatorResponse validateCss(String css) {
        return trackQuery(new JavaMethodWitness() {
        }, ALL_ERRORS_INTERNAL, () -> {
            try {
                log.debug("Validating css: {}", css);
                if (Strings.isNullOrEmpty(css)) {
                    return new TurboCssValidatorResponse(TurboCssValidatorResponse.STATUS_SUCCESS, null, null);
                }

                HttpPost post = new HttpPost(cssValidatorServiceUrl);
                // css
                ObjectNode wrapper = OM.createObjectNode();
                wrapper.set("css", new TextNode(css));
                post.setEntity(new StringEntity(OM.writeValueAsString(wrapper), ContentType.APPLICATION_JSON));

                InputStream content = null;
                try (CloseableHttpResponse response = httpClient.execute(post)) {
                    int statusCode = response.getStatusLine().getStatusCode();
                    content = response.getEntity() == null ? null : response.getEntity().getContent();
                    String contentStr = content == null ? null : IOUtils.toString(content, Charsets.UTF_8);
                    log.debug("Response: {}", contentStr);
                    if (statusCode == HttpStatus.SC_OK && contentStr != null) {
                        log.debug("Validator returns OK for request");
                        // parse result
                        return OM.readValue(contentStr, TurboCssValidatorResponse.class);
                    } else {
                        log.error("Validator returns status code {}", statusCode);
                        throw new WebmasterException("Error from validator service: " + contentStr,
                                new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), contentStr));
                    }
                } finally {
                    if (content != null) {
                        try {
                            content.close();
                        } catch (IOException e) {
                            log.error("Exception when closing stream", e);
                        }
                    }
                }
            } catch (IOException e) {
                throw new WebmasterException("IO error",
                        new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), e.getMessage()), e);
            }
        });
    }

    public static TurboMenuItem validateMenuItem(TurboMenuItem menuItem) throws TurboParserException {
        if (StringUtils.isBlank(menuItem.getLabel())) {
            throw new TurboParserException("Empty menu label", MENU_EMPTY_LABEL);
        }
        if (!Strings.isNullOrEmpty(menuItem.getUrl())) {
            try {
                WebmasterUriUtils.toOldUri(menuItem.getUrl());
            } catch (Exception e) {
                throw new TurboParserException("Invalid menu url", MENU_INVALID_URL);
            }
        }
        return menuItem.withUrl(menuItem.getUrl() == null ? null : menuItem.getUrl().strip());
    }

    public Void validateCheckoutSettings(WebmasterHostId hostId, TurboCommerceSettingsBuilder commerceSettings) throws TurboParserException {
        // проверим url корзины
        String cartUrl = commerceSettings.getCartUrl();
        // проверим принадлежность домену и наличие в path подстановок {offer_id} или {offer_url}
        if (commerceSettings.isCartUrlEnabled()) {
            if (Strings.isNullOrEmpty(cartUrl)) {
                throw new TurboParserException("Empty cart url", INVALID_CART_URL);
            }
            try {
                if (cartUrl.startsWith(TurboConstants.PLACEHOLDER_CART_URL_OFFER_URL)) {
                    // надо, чтобы помимо cartUrl было еще что-нибудь (хотя бы два символа)
                    if ((cartUrl.length() - TurboConstants.PLACEHOLDER_CART_URL_OFFER_URL.length()) < 2) {
                        throw new TurboParserException("Cart url cannot contains only offer_url", INVALID_CART_URL);
                    }
                    // все итак хорошо, больше ничего не проверяем
                } else {
                    if (cartUrl.startsWith("/")) {
                        cartUrl = IdUtils.hostIdToUrl(hostId) + cartUrl;
                    } else {
                        // проверим принадлежность одному домену
                        WebmasterHostId cartHostId = IdUtils.urlToHostId(cartUrl);
                        if (!hostOwnerService.isSameOwner(cartHostId, hostId)) {
                            throw new TurboParserException("Cart url belongs to different host", INVALID_CART_URL);
                        }
                    }
                    // ищем подстановки
                    if (!cartUrl.contains(TurboConstants.PLACEHOLDER_CART_URL_OFFER_ID) &&
                            !cartUrl.contains(TurboConstants.PLACEHOLDER_CART_URL_OFFER_URL)) {
                        throw new TurboParserException("Cart url path does not contains neither {offer_id} nor {offer_url}",
                                INVALID_CART_URL);
                    }
                }
            } catch (Exception e) {
                throw new TurboParserException("Invalid cart url", INVALID_CART_URL);
            }
        }
        // пустой емейл для листингов
        String checkoutEmail = commerceSettings.getCheckoutEmail();
        if (commerceSettings.isTurboCartEnabled() && Strings.isNullOrEmpty(checkoutEmail)) {
            throw new TurboParserException("Empty turbo cart email", EMPTY_TURBOCART_EMAIL);
        }
        if (commerceSettings.isCheckoutEmailEnabled() && Strings.isNullOrEmpty(checkoutEmail)) {
            throw new TurboParserException("Empty checkout email", INVALID_CHECKOUT_EMAIL);
        }
        // проверим емейл для покупок в один клик или турбо-корзины
        if (!Strings.isNullOrEmpty(checkoutEmail)) {
            try {
                checkoutEmail = validateEmail(checkoutEmail, -1);
                commerceSettings.setCheckoutEmail(checkoutEmail);
            } catch (TurboParserException e) {
                throw new TurboParserException("Invalid checkout email", INVALID_CHECKOUT_EMAIL);
            }
        }
        commerceSettings.setCartUrl(cartUrl);
        // минимальная стоимость заказа
        if (commerceSettings.getMinOrderValue().doubleValue() < 0.0) {
            throw new TurboParserException("Invalid min order value", INVALID_MIN_ORDER_VALUE);
        }
        return null;
    }

    public Void validateInfoSections(List<TurboCommerceInfoSection> infoSections) throws TurboParserException {
        if (infoSections != null) {
            if (infoSections.size() > TurboConstants.MAX_COMMERCE_INFO_SECTIONS) {
                throw invalidParamValue("Too many e-commerce info sections", "infoSections", null);
            }
            Set<String> sectionNames = new HashSet<>();
            int index = 0;
            for (TurboCommerceInfoSection infoSection : infoSections) {
                if (Strings.isNullOrEmpty(infoSection.getTitle())) {
                    throw invalidParamValue("Empty e-commerce info section title", "title", "");
                }
                if (Strings.isNullOrEmpty(infoSection.getValue())) {
                    throw invalidParamValue("Empty e-commerce info section value", "value", "");
                }
                if (infoSection.getValue().length() > TurboConstants.MAX_COMMERCE_INFO_SECTION_SIZE) {
                    throw invalidParamValue("Too large e-commerce info section value", "value", "");
                }
                if (sectionNames.contains(infoSection.getTitle())) {
                    throw new TurboParserException("Duplicate e-commerce info section title", DUPLICATE_COMMERCE_SECTION, index);
                }
                sectionNames.add(infoSection.getTitle());
                index++;
            }
        }
        return null;
    }

    public TurboCommerceSettings validateCommerceSettings(WebmasterHostId hostId, TurboCommerceSettings commerceSettings, Long userId,
                                                          Set<TurboHostSettingsBlock> blocks) throws TurboParserException {
        if (commerceSettings == null) {
            return null;
        }
        TurboCommerceSettingsBuilder result = new TurboCommerceSettingsBuilder(commerceSettings);
        if (blocks.contains(TurboHostSettingsBlock.CHECKOUT)) {
            validateCheckoutSettings(hostId, result);
        }
        if (blocks.contains(TurboHostSettingsBlock.INFOSECTIONS)) {
            // проверим инфоблоки
            validateInfoSections(commerceSettings.getInfoSections());
        }
        if (blocks.contains(TurboHostSettingsBlock.DELIVERYSECTION)) {
            //Проверим доставку
            DeliverySection deliverySettings = commerceSettings.getDeliverySection();
            if (deliverySettings != null) {
                deliverySettings.validate("deliverySection");
            }
        }
        if (blocks.contains(TurboHostSettingsBlock.PAYMENTS)) {
            result.setPaymentsSettings(validatePaymentsSettings(WwwUtil.cutWWWAndM(hostId), commerceSettings.getPaymentsSettings(), userId));
        }
        if (blocks.contains(TurboHostSettingsBlock.BITRIX)) {
            result.setBitrixSettings(validateBitrixSettings(WwwUtil.cutWWWAndM(hostId), commerceSettings.getBitrixSettings()));
        }

        return result.build();
    }

    public ServiceMerchantInfo refreshMerchantInfo(String domain, String token, Long userId) {
        if (Strings.isNullOrEmpty(token) && userId == null) {
            return null;
        }
        ServiceMerchantInfo merchantInfo = serviceMerchantCacheYDao.findRecord(domain, token, userId);
        // если уже есть service_merchant_id - обновим его
        if (merchantInfo != null) {
            merchantInfo = paymentsService.getServiceMerchant(merchantInfo.getId());
        } else {
            merchantInfo = paymentsService.createServiceMerchant(domain, token, userId);
        }
        // обновим в БД
        if (merchantInfo != null) {
            serviceMerchantCacheYDao.storeRecord(domain, token, userId, merchantInfo);

        }
        return merchantInfo;
    }

    /**
     * Валидация настроек оплат (платежного токена и/или CheckedUserTicket-а)
     * Пока без CheckedUserTicket
     */
    public TurboPaymentsSettings validatePaymentsSettings(
            String domain, TurboPaymentsSettings paymentsSettings, Long userId) throws TurboParserException {
        if (paymentsSettings == null) {
            return null;
        }
        if (userId != null) {
            // store edit info
            UserPersonalInfo info = userPersonalInfoService.getUserPersonalInfo(userId);
            if (info != null) {
                UserWithLogin editUser = new UserWithLogin(userId, info.getLogin());
                paymentsSettings = paymentsSettings.withEditUser(editUser).withEditDate(DateTime.now());
            }
        }
        String token = paymentsSettings.getToken();
        if (Strings.isNullOrEmpty(token)) {
            return paymentsSettings.toBuilder().editUser(paymentsSettings.getEditUser()).editDate(paymentsSettings.getEditDate()).build();
        }
        // поищем в нашем кэше
        ServiceMerchantInfo merchantInfo = refreshMerchantInfo(domain, token, userId);
        // если уже есть service_merchant_id - обновим его
        if (merchantInfo == null) {
            throw new TurboParserException("Token not found", PAYMENTS_TOKEN_NOT_FOUND);
        }
        /*if (paymentsSettings.isEnabled() && !merchantInfo.isActive()) {
            throw new TurboParserException("Token not confirmed", PAYMENTS_TOKEN_NOT_CONFIRMED);
        }*/
        return paymentsSettings.withMerchantInfo(merchantInfo);
    }

    public TurboBitrixSettings validateBitrixSettings(String domain, TurboBitrixSettings bitrixSettings) throws TurboParserException {
        if (bitrixSettings == null || !bitrixSettings.isEnabled()) {
            return bitrixSettings;
        }
        // проверим, что перед нами корректный URL
        try {
            WebmasterUriUtils.toOldUri(bitrixSettings.getApiBaseUrl());
        } catch (Exception e) {
            throw new TurboParserException("Custom counter value is not valid URL", TurboParserException.ErrorCode.INVALID_BITRIX_URL);
        }

        return bitrixSettings.withEncryptedToken(turboCommerceService.encryptBitrixToken(bitrixSettings.getToken()));
    }

    public TurboFeedbackSettings validateFeedbackSettings(
            WebmasterHostId hostId, TurboFeedbackSettings settings) throws TurboParserException {
        if (settings == null || settings.getButtons() == null) {
            return settings;
        }
        if (settings.getButtons().size() > TurboConstants.MAX_FEEDBACK_BUTTONS) {
            throw invalidParamValue("Too many feedback buttons", "settings",
                    String.valueOf(settings.getButtons().size()));
        }
        for (int i = 0; i < settings.getButtons().size(); i++) {
            settings.getButtons().set(i, validateFeedbackButton(hostId, settings.getButtons().get(i), i));
        }
        return settings;
    }

    public static TurboFeedbackButton validateFeedbackButton(
            WebmasterHostId hostId, TurboFeedbackButton button, int buttonIndex) throws TurboParserException {
        switch (button.getType()) {
            case CALL: // нужна ли проверка номера?
                button = button.withUrl(TurboConstants.PHONE_SCHEME + validatePhoneNumber(button.getUrl(), buttonIndex));
                break;
            case CHAT:
                    /*if (!dialogHostService.isDialogHost(hostId)) {
                        throw invalidParamValue("Host cannot have chat", "type", "chat");
                    }*/
                break;
            case MAIL:
                button = button.withUrl(TurboConstants.EMAIL_SCHEME + validateEmail(button.getUrl(), buttonIndex));
                break;
            case CALLBACK:
                button = button.withUrl(validateEmail(button.getSendTo(), buttonIndex));
                break;
            case FACEBOOK:
            case GOOGLE:
            case ODNOKLASSNIKI:
            case TELEGRAM:
            case TWITTER:
            case VIBER:
            case VKONTAKTE:
            case WHATSAPP:
                // валидируем урл
                try {
                    button = button.withUrl(checkScheme(button.getUrl()));
                } catch (URISyntaxException e) {
                    throw new TurboParserException("Invalid feedback url", INVALID_FEEDBACK_URL, buttonIndex);
                }
                break;
        }
        return button;
    }

    public static String validatePhoneNumber(String phoneNumber, int buttonIndex) throws TurboParserException {
        if (phoneNumber == null) {
            throw new TurboParserException("Empty phone number", INVALID_FEEDBACK_URL, buttonIndex);
        }
        phoneNumber = phoneNumber.trim();
        if (phoneNumber.startsWith(TurboConstants.PHONE_SCHEME)) {
            phoneNumber = phoneNumber.substring(TurboConstants.PHONE_SCHEME.length()).trim();
        }
        if (phoneNumber.isEmpty()) {
            throw new TurboParserException("Empty phone number", INVALID_FEEDBACK_URL, buttonIndex);
        }
        return phoneNumber;
    }

    public static String validateEmail(String email, int buttonIndex) throws TurboParserException {
        // проверяем только домен
        if (email == null) {
            throw invalidParamValue("Empty email", "url", null);
        }
        email = email.trim();
        if (email.startsWith(TurboConstants.EMAIL_SCHEME)) {
            email = email.substring(TurboConstants.EMAIL_SCHEME.length()).trim();
        }
        if (!EmailValidator.isValid(email)) {
            throw new TurboParserException("Invalid email pattern", INVALID_FEEDBACK_EMAIL, buttonIndex);
        }
        // найдем домен
        int atIndex = email.indexOf('@');
        if (atIndex <= 0 || atIndex >= email.length() || email.indexOf('@', atIndex + 1) != -1) {
            throw new TurboParserException("Invalid feedback email", INVALID_FEEDBACK_EMAIL, buttonIndex);
        }
        String emailDomain = email.substring(atIndex + 1);
        try {
            // домен не должен начинаться со схемы
            if (emailDomain.startsWith(Schema.HTTP.getSchemaPrefix()) ||
                    emailDomain.startsWith(Schema.HTTPS.getSchemaPrefix())) {
                throw new TurboParserException("Invalid feedback email domain", INVALID_FEEDBACK_EMAIL, buttonIndex);
            }
        } catch (WebmasterException | IllegalArgumentException e) {
            throw new TurboParserException("Invalid feedback email domain", INVALID_FEEDBACK_EMAIL, buttonIndex);
        }
        return email;
    }

    public TurboUserAgreement validateUserAgreement(TurboUserAgreement userAgreement) throws TurboParserException {
        if (userAgreement != null) {
            String agreementLink = userAgreement.getAgreementLink();
            if (Strings.isNullOrEmpty(userAgreement.getAgreementCompany()) ^ Strings.isNullOrEmpty(
                    agreementLink)) {
                throw new TurboParserException("Invalid agreement settings", INVALID_FEEDBACK_AGREEMENT, -1);
            }
            try {
                if (!Strings.isNullOrEmpty(agreementLink)) {
                    WebmasterUriUtils.toOldUri(agreementLink); // исключительно для проверки
                    // TODO сделать костыль покрасивее
                    if (!agreementLink.contains("://")) {
                        agreementLink = "http://" + agreementLink;
                    }
                    userAgreement = userAgreement.withAgreementLink(agreementLink);
                }
            } catch (Exception e) {
                throw new TurboParserException("Invalid agreement link url", INVALID_FEEDBACK_AGREEMENT_URL, -1);
            }
        }
        return userAgreement;
    }

    private static WebmasterException invalidParamValue(String message, String param, String value) {
        return new WebmasterException(message,
                new WebmasterErrorResponse.IllegalParameterValueResponse(TurboParserService.class, param, value));
    }

    static String checkScheme(String url) throws URISyntaxException {
        URI uri = new URI(url);
        if (uri.getScheme() == null) {
            return "http:" + (url.startsWith("//") ? "" : "//") + url;
        } else if (!SUPPORTED_SCHEMES.contains(uri.getScheme())) {
            throw new URISyntaxException(url, "Unsupported scheme: " + uri.getScheme());
        }
        return uri.toString();
    }

    /**
     * Адаптация кода https://a.yandex-team.ru/arc/trunk/arcadia/quality/functionality/turbo/merger/lib/common.cpp?rev=7458491#L172 для джавы
     */
    private static String сorrectCartUrlCgiDelimeters(String cartUrl, String offerUrl) {
        char[] chars = cartUrl.toCharArray();
        String placeholder = TurboConstants.PLACEHOLDER_CART_URL_OFFER_URL;
        int placeholderLength = placeholder.length();
        if (cartUrl.startsWith(placeholder) && chars.length > placeholderLength &&
                (chars[placeholderLength] == '&' || chars[placeholderLength] == '?')
        ) {
            int qPos = offerUrl.indexOf('?');
            if (qPos == -1) {
                if (chars[placeholderLength] == '&') {
                    chars[placeholderLength] = '?';
                    return new String(chars);
                }
            } else {
                if (qPos == offerUrl.length() - 1 || offerUrl.endsWith("&")) {
                    ArrayUtils.remove(chars, placeholderLength);
                    return new String(chars);
                } else if (chars[placeholderLength] == '?') {
                    chars[placeholderLength] = '&';
                    return new String(chars);
                }
            }
        }
        return cartUrl;
    }

    /**
     * Валидация ссылки перехода в корзину (WMC-6482)
     *
     * @param hostId
     * @param cartUrl
     * @return
     */
    public CartUrlValidationResult validateCartUrl(WebmasterHostId hostId, String cartUrl, String offerXml)
            throws Exception {
        log.info("Validating cartUrl {} for host {}", cartUrl, hostId);
        // читаем наш xml
        Document document = TurboXMLReader.parseTurboFeed(new ByteArrayInputStream(offerXml.getBytes()), 0, null, 0);
        // вытаскиваем нужные значения из оффера
        String offerId = offerIdExpr.evaluate(document);
        String offerUrl = offerUrlExpr.evaluate(document);
        String offerName = offerNameExpr.evaluate(document);
        log.info("Extracted offer parameters: id = {}, url = {}, name = {}", offerId, offerUrl, offerName);
        // формируем ссылку
        String resultingCartUrl = сorrectCartUrlCgiDelimeters(cartUrl, offerUrl)
                .replaceAll(Pattern.quote(TurboConstants.PLACEHOLDER_CART_URL_OFFER_ID), offerId)
                .replaceAll(Pattern.quote(TurboConstants.PLACEHOLDER_CART_URL_OFFER_URL), offerUrl);
        // качаем документ по ссылке
        log.debug("Resulting url: {}", resultingCartUrl);
        // заэнкоженный offerUrl
        String encodedOfferUrl = offerUrl, offerPath = null;
        try {
            URI uri = new URI(offerUrl);
            // будем искать только relative url, path + query
            offerPath = Optional.ofNullable(uri.getPath()).orElse("/");
            if (offerPath.endsWith("/")) {
                offerPath = offerPath.substring(0, offerPath.length() - 1);
            }
            offerUrl = Optional.ofNullable(uri.getPath()).orElse("/") +
                    Optional.ofNullable(uri.getQuery()).map(q -> "?" + q).orElse("");
            encodedOfferUrl = Optional.ofNullable(uri.getRawPath()).orElse("/") +
                    Optional.ofNullable(uri.getRawQuery()).map(q -> "?" + q).orElse("");
            log.info("Offer url for search: {}, encoded: {}", offerUrl, encodedOfferUrl);

        } catch (URISyntaxException e) {
            log.warn("Offer url is not valid: {}", offerUrl);
        }
        try {
            String cartPage = zoraForValidatorsService.processEntityGoZora(resultingCartUrl, EntityUtils::toString);
            log.debug("Resulting page: ");
            log.debug("=========================================================================");
            log.debug(cartPage);
            log.debug("=========================================================================");
            // наша чудо-эвристика
            // TODO ignoreCase ?
            boolean containsOfferName = cartPage.contains(offerName);
            boolean containsOfferUrl = offerUrl.length() > 1 &&
                    (cartPage.contains(offerUrl) || cartPage.contains(encodedOfferUrl));
            boolean hasJavascriptRedirect = cartPage.contains("location.href");
            boolean containsOfferPath = offerPath != null && cartPage.contains(offerPath);

            CartUrlValidationResult result = new CartUrlValidationResult(resultingCartUrl,
                    new HttpCodeInfo(HttpStatus.SC_OK), containsOfferName, containsOfferUrl, hasJavascriptRedirect,
                    containsOfferPath);
            log.info("Validation cartUrl {} for host {} result: {}", resultingCartUrl, hostId, result);
            return result;
        } catch (WebmasterException e) {
            log.error("Error when validation cartUrl " + resultingCartUrl + " for host " + hostId, e);
            if (e.getError() instanceof SitaErrorResponse) {
                Integer httpCode = ((SitaErrorResponse) e.getError()).getHttpCode();
                return new CartUrlValidationResult(resultingCartUrl, new HttpCodeInfo(httpCode), false, false, false,
                        false);
            }
            throw e;
        }
    }

    public static TurboSearchSettings validateSearchSettings(WebmasterHostId hostId, TurboSearchSettings settings)
            throws TurboParserException {
        // only url check
        try {
            String url = settings.getUrl();
            if (Strings.isNullOrEmpty(url)) {
                throw new TurboParserException("Empty search url", INVALID_SEARCH_URL);
            }
            if (url.startsWith("/")) {
                url = IdUtils.hostIdToUrl(hostId) + url;
            }
            WebmasterHostId urlHostId = IdUtils.urlToHostId(url);
            if (!WwwUtil.cutWWWAndM(urlHostId).equals(WwwUtil.cutWWWAndM(hostId))) {
                throw new TurboParserException("Search url belongs to other host", INVALID_SEARCH_URL);
            }
            if (!url.contains(PLACEHOLDER_SEARCH_URL_TEXT)) {
                throw new TurboParserException("Search url does not contains {text} placeholder", INVALID_SEARCH_URL);
            }
            if (!url.startsWith(Schema.HTTP.getSchemaPrefix()) && !url.startsWith(Schema.HTTPS.getSchemaPrefix())) {
                url = hostId.getSchema().getSchemaPrefix() + url;
            }
            settings = new TurboSearchSettings(url, settings.getCharset(), settings.getPlaceholder());
            return settings;
        } catch (Exception e) {
            throw new TurboParserException("Invalid search url", e, INVALID_SEARCH_URL);
        }
    }

    @Override
    public void deleteUserData(WebmasterUser user) {
        serviceMerchantCacheYDao.deleteForUser(user.getUserId());
    }

    @Override
    public @NotNull List<String> getTakeoutTables() {
        return List.of(
                serviceMerchantCacheYDao.getTablePath()
        );
    }

    @lombok.Value
    public static class MediaSize implements Comparable<MediaSize> {
        String val;
        long height;
        long width;

        @Override
        public int compareTo(@NotNull TurboParserService.MediaSize o) {
            int val = Long.compare(height * width, o.height * o.width);
            return val == 0 ? Long.compare(height, o.height) : val;
        }
    }

}
