package ru.yandex.webmaster3.storage.ugc;

import java.util.List;
import java.util.function.Consumer;

import lombok.Builder;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.Nullable;
import org.joda.time.DateTime;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.ValueDataMapper;

@Slf4j
@Repository
public class SiteRatingCacheYDao extends AbstractYDao {
   private static final DataMapper<CacheSiteRating> MAPPER = DataMapper.create(
           F.DOMAIN,
           F.EXIST,
           F.POSITIVE_REVIEWS_COUNT,
           F.NEGATIVE_REVIEWS_COUNT,
           F.COMMENTS_COUNT,
           F.TTL_DATE,
           SiteRatingCacheYDao::mapper
   );

   protected SiteRatingCacheYDao() {
       super(PREFIX_CACHE, "site_rating_cache");
   }

   public void insertEmpty(String domain) {
       upsert(
               F.DOMAIN.value(domain),
               F.EXIST.value(false),
               F.TTL_DATE.value(DateTime.now())
       ).execute();
   }

   @Nullable
   public CacheSiteRating select(String domain) {
       return select(MAPPER)
               .where(F.DOMAIN.eq(domain))
               .queryOne();
   }

   public void insert(String domain, Long positiveReviewsCount, Long negativeReviewsCount, Long commentsCount) {
       upsert(
               F.DOMAIN.value(domain),
               F.EXIST.value(true),
               F.POSITIVE_REVIEWS_COUNT.value(positiveReviewsCount),
               F.NEGATIVE_REVIEWS_COUNT.value(negativeReviewsCount),
               F.COMMENTS_COUNT.value(commentsCount),
               F.TTL_DATE.value(DateTime.now())
       ).execute();
   }

   private static CacheSiteRating mapper(String domain, boolean exist, Long positiveReviewsCount, Long negativeReviewsCount, Long commentsCount, DateTime ttlDate) {
       if (!exist) {
           return CacheSiteRating.builder()
                   .domain(domain)
                   .exist(false)
                   .ttlDate(ttlDate)
                   .build();
       }

       return CacheSiteRating.builder()
               .domain(domain)
               .exist(true)
               .ttlDate(ttlDate)
               .siteRating(SiteRating.build(
                       positiveReviewsCount,
                       negativeReviewsCount,
                       commentsCount
               ))
               .build();
   }

    public void batchInsert(List<CacheSiteRating> items) {
       batchInsert(VALUE_MAPPER, items).execute();
    }

    public void forEach(Consumer<CacheSiteRating> consumer) {
        streamReader(MAPPER, consumer);
    }

    static final ValueDataMapper<CacheSiteRating> VALUE_MAPPER = ValueDataMapper.create(
            Pair.of(F.DOMAIN, csr -> F.DOMAIN.get(csr.getDomain())),
            Pair.of(F.EXIST, csr -> F.EXIST.get(csr.isExist())),
            Pair.of(F.POSITIVE_REVIEWS_COUNT, csr -> F.POSITIVE_REVIEWS_COUNT.get((!csr.isExist()) ? null : csr.getSiteRating().getPositiveReviewsCount())),
            Pair.of(F.NEGATIVE_REVIEWS_COUNT, csr -> F.NEGATIVE_REVIEWS_COUNT.get((!csr.isExist()) ? null : csr.getSiteRating().getNegativeReviewsCount())),
            Pair.of(F.COMMENTS_COUNT, csr -> F.COMMENTS_COUNT.get((!csr.isExist()) ? null : csr.getSiteRating().getCommentsCount())),
            Pair.of(F.TTL_DATE, csr -> F.TTL_DATE.get(csr.getTtlDate()))
    );

   @Data
   @Builder
   public static class CacheSiteRating {
       boolean exist;
       String domain;
       @Nullable
       SiteRating siteRating;
       DateTime ttlDate;
   }

   private interface F {
       Field<String> DOMAIN = Fields.stringField("domain");
       Field<Boolean> EXIST = Fields.boolField("exist");
       Field<Long> POSITIVE_REVIEWS_COUNT = Fields.longField("positive_reviews_count").makeOptional();
       Field<Long> NEGATIVE_REVIEWS_COUNT = Fields.longField("negative_reviews_count").makeOptional();
       Field<Long> COMMENTS_COUNT = Fields.longField("comments_count").makeOptional();
       Field<DateTime> TTL_DATE = Fields.jodaDateTimeField("ttl_date");
   }
}
