package ru.yandex.webmaster3.storage.ugc;

import java.io.IOException;
import java.net.URI;
import java.util.Optional;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import com.Ostermiller.util.Base64;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.JsonNode;
import lombok.Setter;
import lombok.Value;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.config.SocketConfig;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.HttpConstants;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.metrics.externals.AbstractExternalAPIService;
import ru.yandex.webmaster3.core.util.JavaMethodWitness;
import ru.yandex.webmaster3.core.util.json.JsonMapping;

/**
 * author: ishalaru
 * DATE: 12.08.2019
 */
@Service("ugcIntegrationService")
public class UGCIntegrationService extends AbstractExternalAPIService {
    private static final Logger log = LoggerFactory.getLogger(UGCIntegrationService.class);
    private static final int SOCKET_TIMEOUT = 2_000;

    @org.springframework.beans.factory.annotation.Value("${webmaster3.storage.ugc.url}")
    @Setter
    private String url;
    @org.springframework.beans.factory.annotation.Value("${webmaster3.storage.ugc.appId}")
    @Setter
    private String appId;


    @org.springframework.beans.factory.annotation.Value("${webmaster3.storage.ugc.review.url}")
    @Setter
    private String reviewUrl;
    @org.springframework.beans.factory.annotation.Value("${webmaster3.storage.ugc.review.appId}")
    @Setter
    private String reviewAppId;


    private CloseableHttpClient httpClient;

    @PostConstruct
    public void init() {
        SocketConfig socketConfig = SocketConfig.custom()
                .setSoTimeout(SOCKET_TIMEOUT)
                .build();

        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectTimeout(HttpConstants.DEFAULT_CONNECT_TIMEOUT)
                .setSocketTimeout(SOCKET_TIMEOUT)
                .setConnectionRequestTimeout(HttpConstants.DEFAULT_CONNECTION_REQUEST_TIMEOUT)
                .build();

        httpClient = HttpClientBuilder.create()
                .setDefaultSocketConfig(socketConfig)
                .setMaxConnPerRoute(5)
                .setMaxConnTotal(5)
                .setDefaultRequestConfig(requestConfig)
                .build();
    }

    @PreDestroy
    public void destroy() throws IOException {
        httpClient.close();
    }

    private Optional<byte[]> runQuery(URI uri) {
        return trackQuery(new JavaMethodWitness() {
        }, ALL_ERRORS_INTERNAL, () -> {
            HttpGet request = new HttpGet(uri);
            log.info("Querying ugc with {}", request);
            try (CloseableHttpResponse response = httpClient.execute(request)) {
                int httpCode = response.getStatusLine().getStatusCode();
                if (httpCode / 100 == 2) {
                    return Optional.of(response.getEntity().getContent().readAllBytes());
                } else {
                    return Optional.empty();
                }
            } catch (IOException e) {
                log.error(e.getMessage(), e);
                return null;
            }
        });
    }

    private URI buildDigestUri(final String reviewId) {
        try {
            URIBuilder uriBuilder = new URIBuilder(url);
            uriBuilder.setPath("/ugc/review-info");
            uriBuilder.addParameter("app_id", appId);
            uriBuilder.addParameter("otype", "Site");
            uriBuilder.addParameter("review_id", reviewId);
            return uriBuilder.build();
        } catch (Exception exp) {
            throw new WebmasterException("can't parse uri",
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), exp.getMessage()));
        }
    }

    public UGCReview loadFromDigest(String reviewId) {
        URI buildUri = buildDigestUri(reviewId);
        final Optional<byte[]> data = runQuery(buildUri);
        try {
            return JsonMapping.OM.readValue(data.get(), UGCReview.class);
        } catch (IOException e) {
            return null;
        }
    }

    private URI buildThermometrUri(final String domain) {
        try {
            URIBuilder uriBuilder = new URIBuilder(reviewUrl);
            uriBuilder.setPath("/ugcpub/object-digest");
            uriBuilder.addParameter("json", "1");
            uriBuilder.addParameter("object", "/site/" + Base64.encode(domain));
            uriBuilder.addParameter("app_id", reviewAppId);
            uriBuilder.addParameter("otype", "Site");
            return uriBuilder.build();
        } catch (Exception exp) {
            throw new WebmasterException("can't parse uri",
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), exp.getMessage()));
        }
    }

    public SiteRating loadUGCThermometrInfo(String domain) {

        URI buildUri = buildThermometrUri(domain);
        final Optional<byte[]> data = runQuery(buildUri);
        return data.map(e -> {
            try {
                JsonNode root = JsonMapping.OM.readTree(e);
                return parseSiteRating(root.get("digest"));
            } catch (IOException ioException) {
                return null;
            }

        }).orElse(null);
    }

    private SiteRating parseSiteRating(JsonNode counter) {
        if (!counter.has("positiveReviewsCount") || !counter.has("negativeReviewsCount")
        || !counter.has("object") || !counter.get("object").has("ratingCount")) {
            return null;
        }

        return SiteRating.build(
                counter.get("positiveReviewsCount").asLong(),
                counter.get("negativeReviewsCount").asLong(),
                counter.get("object").get("ratingCount").asLong(0L)
        );
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    @Value
    public static class UGCReview {
        @JsonProperty("id")
        String id;
        @JsonProperty("time")
        DateTime time;
        @JsonProperty("text")
        String text;
        @JsonProperty("author")
        Author author;
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    @Value
    public static class Author {
        @JsonProperty("id")
        String id;
        @JsonProperty("name")
        String name;
    }

}
