package ru.yandex.webmaster3.storage.url.checker3.dao;

import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.springframework.stereotype.Repository;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.url.checker3.UrlCheckDeviceType;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;

import java.util.List;
import java.util.UUID;


/**
 * @author leonidrom
 */
@Repository
public class UrlCheckRequests2YDao extends AbstractYDao  {
    private static final String CREATED_TIME_INDEX = "created_time_index";

    public UrlCheckRequests2YDao() {
        super(PREFIX_URLCHECK, "url_check_requests2");
    }

    public long getHostRequestsCountAfter(WebmasterHostId hostId, DateTime fromDate) {
        return countAll()
                .secondaryIndex(CREATED_TIME_INDEX)
                .where(F.HOST_ID.eq(hostId))
                .and(F.CREATED_TIME.gte(fromDate))
                .queryOne();
    }

    public List<Record> getRequests(WebmasterHostId hostId) {
        return select(MAPPER).where(F.HOST_ID.eq(hostId)).queryForList(Pair.of(F.REQUEST_ID, Record::requestId));
    }

    public Record getRequest(UUID requestId) {
        return select(MAPPER)
                .where(F.REQUEST_ID.eq(requestId))
                .queryOne();
    }

    public Record storeRequest(WebmasterHostId hostId, UUID requestId, String url, UrlCheckDeviceType deviceType,
                               DateTime createdTime) {
        Record rec = new Record(hostId, requestId, url, deviceType, createdTime);
        storeRequest(rec);

        return rec;
    }

    public void storeRequest(Record rec) {
        insert(
                F.HOST_ID.value(rec.hostId),
                F.REQUEST_ID.value(rec.requestId),
                F.URL.value(rec.url),
                F.DEVICE_TYPE.value(rec.deviceType),
                F.CREATED_TIME.value(rec.createdTime)
        ).execute();
    }

    public void deleteRequest(WebmasterHostId hostId, UUID requestId) {
        delete()
                .where(F.HOST_ID.eq(hostId))
                .and(F.REQUEST_ID.eq(requestId))
                .execute();
    }

    protected static final class F {
        public static final Field<WebmasterHostId> HOST_ID = Fields.hostIdField("host_id");
        public static final Field<UUID> REQUEST_ID = Fields.uuidField("request_id");
        public static final Field<String> URL = Fields.stringField("url");
        public static final Field<UrlCheckDeviceType> DEVICE_TYPE = Fields.stringEnumField("device_type", UrlCheckDeviceType.R);
        public static final Field<DateTime> CREATED_TIME = Fields.jodaDateTimeField("created_time");
    }

    protected static final DataMapper<Record> MAPPER = DataMapper.create(
            F.HOST_ID, F.REQUEST_ID, F.URL, F.DEVICE_TYPE, F.CREATED_TIME, Record::new
    );

    public record Record(WebmasterHostId hostId, UUID requestId, String url, UrlCheckDeviceType deviceType,
                         DateTime createdTime) {};
}
