package ru.yandex.webmaster3.storage.url.checker3.service;

import NUrlChecker.Request;
import NUrlChecker.Response;
import com.datastax.driver.core.utils.UUIDs;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.logbroker.writer.LogbrokerClient;
import ru.yandex.webmaster3.core.solomon.SolomonSensor;
import ru.yandex.webmaster3.core.solomon.metric.SolomonKey;
import ru.yandex.webmaster3.core.solomon.metric.SolomonMetricRegistry;
import ru.yandex.webmaster3.core.solomon.metric.SolomonTimer;
import ru.yandex.webmaster3.core.solomon.metric.SolomonTimerConfiguration;
import ru.yandex.webmaster3.core.url.checker3.UrlCheckRequestParams;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.RetryUtils;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.core.worker.client.WorkerClient;
import ru.yandex.webmaster3.storage.crawl.RotorSettings;
import ru.yandex.webmaster3.storage.crawl.dao.RotorSettingsYDao;
import ru.yandex.webmaster3.storage.jupiter.JupiterUtils;
import ru.yandex.webmaster3.storage.url.checker3.dao.UrlCheckDataBlocksYDao;
import ru.yandex.webmaster3.storage.url.checker3.dao.UrlCheckRequests2YDao;
import ru.yandex.webmaster3.storage.url.checker3.data.FetchUrlCheckDataBlockTaskData;
import ru.yandex.webmaster3.storage.url.checker3.data.UrlCheckDataBlock;
import ru.yandex.webmaster3.storage.url.checker3.data.UrlCheckDataBlockState;
import ru.yandex.webmaster3.storage.url.checker3.data.UrlCheckDataBlockType;
import ru.yandex.webmaster3.storage.url.checker3.data.blocks.RotorRenderSettingsData;
import ru.yandex.webmaster3.storage.url.checker3.data.blocks.SearchBaseDateData;
import ru.yandex.webmaster3.storage.url.checker3.data.blocks.UrlCheckRequestParamsData;
import ru.yandex.webmaster3.storage.url.common.data.UrlCheckRequestSource;

import javax.annotation.PostConstruct;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Stream;

/**
 * @author leonidrom
 */
@Slf4j
@Service
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class UrlCheckDataBlocksService {
    private static final RetryUtils.RetryPolicy RETRY_POLICY = RetryUtils.instantRetry(3);
    private static final RetryUtils.RetryPolicy LB_WRITE_RETRY_POLICY = RetryUtils.linearBackoff(5, Duration.standardSeconds(30));

    private static final List<UrlCheckDataBlockType> ROTOR_CHECK_DATA_BLOCKS_RENDER_ON = List.of(
            UrlCheckDataBlockType.ROBOT_ROTOR_CHECK_RENDER_ON,
            UrlCheckDataBlockType.ROBOT_SERVER_RESPONSE_RENDER_ON,
            UrlCheckDataBlockType.ROBOT_ARCHIVE_RENDER_ON
    );

    private static final List<UrlCheckDataBlockType> ROTOR_CHECK_DATA_BLOCKS_RENDER_OFF = List.of(
            UrlCheckDataBlockType.ROBOT_ROTOR_CHECK_RENDER_OFF,
            UrlCheckDataBlockType.ROBOT_SERVER_RESPONSE_RENDER_OFF,
            UrlCheckDataBlockType.ROBOT_ARCHIVE_RENDER_OFF
    );

    private static final List<UrlCheckDataBlockType> ROTOR_CHECK_DATA_BLOCKS = Stream.of(
                ROTOR_CHECK_DATA_BLOCKS_RENDER_ON,
                ROTOR_CHECK_DATA_BLOCKS_RENDER_OFF,
                List.of(UrlCheckDataBlockType.URL_CHECK_REQUEST_PARAMS))
            .flatMap(Collection::stream)
            .toList();

    private static final List<UrlCheckDataBlockType> URL_CHECK_DATA_BLOCKS = List.of(
            UrlCheckDataBlockType.URL_CHECK_REQUEST_PARAMS,
            UrlCheckDataBlockType.SEARCH_BASE_DATE,
            UrlCheckDataBlockType.SEARCH_INFO,
            //UrlCheckDataBlockType.MOBILE_AUDIT,
            UrlCheckDataBlockType.ROTOR_RENDER_SETTINGS,
            UrlCheckDataBlockType.TEXT_CONTENT,
            UrlCheckDataBlockType.ROBOT_INDEXING_INFO
    );

    private final LogbrokerClient urlCheckLogbrokerClient;
    private final UrlCheckDataBlocksYDao urlCheckDataBlocksYDao;
    private final RotorSettingsYDao rotorSettingsYDao;
    private final UrlCheckRequests2YDao urlCheckRequests2YDao;
    private final JupiterUtils jupiterUtils;
    private final WorkerClient workerClient;

    private final SolomonMetricRegistry solomonMetricRegistry;
    private final SolomonTimerConfiguration urlCheckDataBlocksTimerConfiguration;
    private final Map<Pair<UrlCheckDataBlockType, UrlCheckDataBlockState>, SolomonTimer> timers = new ConcurrentHashMap<>();

    @PostConstruct
    public void init() {
        for (var blockType : UrlCheckDataBlockType.values()) {
            for (var blockState : UrlCheckDataBlockState.values()) {
                getCachedTimer(blockType, blockState);
            }
        }
    }

    public UUID sendRotorCheckRequest(UrlCheckRequestParams urlCheckParams) {
        UUID requestId = UUIDs.timeBased();

        var req = Request.TUrlCheckRequest.newBuilder()
                .setTargetDevice(urlCheckParams.getDeviceType().getRobotDeviceType())
                .setUrl(urlCheckParams.getUrl())
                .setDataOrder(getRobotBlocks(ROTOR_CHECK_DATA_BLOCKS))
                .setId(requestId.toString())
                .setViewportWidth(urlCheckParams.getScreenshotWidth())
                .setViewportHeight(urlCheckParams.getScreenshotHeight())
                .setTimestamp((int)(System.currentTimeMillis() / 1000))
                .build();

        urlCheckDataBlocksYDao.add(requestId, ROTOR_CHECK_DATA_BLOCKS, UrlCheckRequestSource.ROTOR_CHECK, DateTime.now());
        closeRequest(requestId, UrlCheckDataBlockType.URL_CHECK_REQUEST_PARAMS,
                JsonMapping.writeValueAsString(new UrlCheckRequestParamsData(urlCheckParams)));

        try {
            RetryUtils.execute(LB_WRITE_RETRY_POLICY, () -> urlCheckLogbrokerClient.write(req.toByteArray()));
        } catch (Exception e) {
            throw new WebmasterException("Failed to write to LB",
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), "Failed to write to LB"), e);
        }

        return requestId;
    }

    public UUID sendUrlCheckRequest(UrlCheckRequestParams urlCheckParams) {
        var hostId = IdUtils.urlToHostId(urlCheckParams.getUrl());
        RotorSettings.RenderSettings renderSettings = RotorSettings.RenderSettings.RENDER_AUTO;
        var rotorSettings = rotorSettingsYDao.getSettings(hostId);
        if (rotorSettings != null) {
            renderSettings = rotorSettings.renderSettings();
        }

        var blockTypes = new ArrayList<>(URL_CHECK_DATA_BLOCKS);
        switch (renderSettings) {
            case RENDER_ON -> blockTypes.addAll(ROTOR_CHECK_DATA_BLOCKS_RENDER_ON);
            case RENDER_OFF, RENDER_AUTO -> {
                renderSettings = RotorSettings.RenderSettings.RENDER_OFF;
                blockTypes.addAll(ROTOR_CHECK_DATA_BLOCKS_RENDER_OFF);
            }
        }

        var requestId = doSendUrlCheckRequest(blockTypes, urlCheckParams, renderSettings);
        urlCheckRequests2YDao.storeRequest(hostId, requestId, urlCheckParams.getUrl(), urlCheckParams.getDeviceType(),
                DateTime.now());

        return requestId;
    }

    public UUID sendUrlCheckRequestNoHost(UrlCheckRequestParams urlCheckParams) {
        RotorSettings.RenderSettings renderSettings = RotorSettings.RenderSettings.RENDER_OFF;
        var blockTypes = new ArrayList<>(ROTOR_CHECK_DATA_BLOCKS_RENDER_ON);
        blockTypes.addAll(URL_CHECK_DATA_BLOCKS);

        return doSendUrlCheckRequest(blockTypes, urlCheckParams, renderSettings);
    }

    private UUID doSendUrlCheckRequest(Collection<UrlCheckDataBlockType> blockTypes,
                                       UrlCheckRequestParams urlCheckParams,
                                       RotorSettings.RenderSettings renderSettings) {
        UUID requestId = UUIDs.timeBased();

        // добавим в базу записи про все блоки данных, которые нам нужны для этого запроса
        urlCheckDataBlocksYDao.add(requestId, blockTypes, UrlCheckRequestSource.URL_CHECK, DateTime.now());

        // сразу же добавим данные для некоторых из блоков
        DateTime curBaseDate = new DateTime(jupiterUtils.getCurrentBaseCollectionDate());
        closeRequest(requestId, UrlCheckDataBlockType.SEARCH_BASE_DATE,
                JsonMapping.writeValueAsString(new SearchBaseDateData(curBaseDate)));

        closeRequest(requestId, UrlCheckDataBlockType.ROTOR_RENDER_SETTINGS,
                JsonMapping.writeValueAsString(new RotorRenderSettingsData(renderSettings)));

        closeRequest(requestId, UrlCheckDataBlockType.URL_CHECK_REQUEST_PARAMS,
                JsonMapping.writeValueAsString(new UrlCheckRequestParamsData(urlCheckParams)));

        // отправим запрос в Роботный сервис
        var robotReqB = Request.TUrlCheckRequest.newBuilder()
                .setTargetDevice(urlCheckParams.getDeviceType().getRobotDeviceType())
                .setUrl(urlCheckParams.getUrl())
                .setDataOrder(getRobotBlocks(blockTypes))
                .setId(requestId.toString())
                .setViewportWidth(urlCheckParams.getScreenshotWidth())
                .setViewportHeight(urlCheckParams.getScreenshotHeight())
                .setUserAgent(urlCheckParams.getUserAgent().getValue())
                .setTimestamp((int)(System.currentTimeMillis() / 1000));

        if (urlCheckParams.getIfModifiedSince() != null) {
            robotReqB.setIfModifiedSince((int)(urlCheckParams.getIfModifiedSince().getMillis() / 1000));
        }

        try {
            urlCheckLogbrokerClient.write(robotReqB.build().toByteArray());
        } catch (Exception e) {
            throw new WebmasterException("Failed to write to LB",
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), "Failed to write to LB"), e);
        }

        // создадим воркерные таски для блоков данных, данные которых достаются на нашей стороне
        var hostId = IdUtils.urlToHostId(urlCheckParams.getUrl());
        var td1 = new FetchUrlCheckDataBlockTaskData(hostId, requestId, UrlCheckDataBlockType.SEARCH_INFO,
                curBaseDate, urlCheckParams);
        var td2 = new FetchUrlCheckDataBlockTaskData(hostId, requestId, UrlCheckDataBlockType.TEXT_CONTENT,
                curBaseDate, urlCheckParams);
        workerClient.enqueueBatch(List.of(td1, td2));

        return requestId;
    }

    @NotNull
    private Integer getRobotBlocks(Collection<UrlCheckDataBlockType> blockTypes) {
        return blockTypes.stream()
                .map(UrlCheckDataBlockType::getRobotItem)
                .filter(Objects::nonNull)
                .map(Request.EDataOrderItem::getNumber)
                .reduce(0, (v1, v2) -> v1 | v2);
    }

    public void closeRequest(UUID requestId, UrlCheckDataBlockType blockType, UrlCheckDataBlockState fetchState) {
        try {
            RetryUtils.execute(RETRY_POLICY, () -> urlCheckDataBlocksYDao.closeRequest(requestId, blockType, fetchState));
            updateMetrics(requestId, blockType, fetchState);
        } catch (Exception e) {
            closeRequestInternalError(requestId, blockType, e);
        }
    }

    public void closeRequest(UUID requestId, UrlCheckDataBlockType blockType, String data) {
        try {
            RetryUtils.execute(RETRY_POLICY, () -> urlCheckDataBlocksYDao.closeRequest(requestId, blockType, data));
            updateMetrics(requestId, blockType, UrlCheckDataBlockState.DONE);
        } catch (Exception e) {
            closeRequestInternalError(requestId, blockType, e);
        }
    }

    public void closeRequestInternalError(UUID requestId, UrlCheckDataBlockType blockType, Exception e) {
        log.error("Block fetch exception: {}, {}", requestId, blockType, e);
        String error = ExceptionUtils.getStackTrace(e);

        try {
            RetryUtils.execute(RETRY_POLICY, () -> urlCheckDataBlocksYDao.closeRequestWithError(requestId, blockType,
                    UrlCheckDataBlockState.INTERNAL_ERROR, error));
        } catch (Exception ex) {
            log.error("Failed to update YDB", ex);
        } finally {
            updateMetrics(requestId, blockType, UrlCheckDataBlockState.INTERNAL_ERROR);
        }
    }

    public void closeRequestRobotInternalError(UUID requestId, UrlCheckDataBlockType blockType,
                                               Response.TUrlCheckResponse.EStatusCode statusCode) {
        log.error("Robot block fetch failure: {}, {}, {}", requestId, blockType, statusCode);
        try {
            RetryUtils.execute(RETRY_POLICY, () -> urlCheckDataBlocksYDao.closeRequestWithError(requestId, blockType,
                    UrlCheckDataBlockState.ROBOT_FETCH_ERROR, statusCode.toString()));
        } catch (Exception ex) {
            log.error("Failed to update YDB", ex);
        } finally {
            updateMetrics(requestId, blockType, UrlCheckDataBlockState.ROBOT_FETCH_ERROR);
        }
    }

    public List<UrlCheckDataBlock> get(UUID requestId) {
        return urlCheckDataBlocksYDao.get(requestId);
    }

    public UrlCheckDataBlock getWithoutData(UUID requestId, UrlCheckDataBlockType blockType) {
        return urlCheckDataBlocksYDao.getWithoutData(requestId, blockType);
    }

    public List<UrlCheckDataBlock> getWithoutData(UUID requestId) {
        return urlCheckDataBlocksYDao.getWithoutData(requestId);
    }

    public List<UrlCheckDataBlock> getWithoutData(DateTime fromDate, DateTime toDate) {
        return urlCheckDataBlocksYDao.getWithoutData(fromDate, toDate);
    }

    private SolomonTimer createTimer(UrlCheckDataBlockType blockType, UrlCheckDataBlockState blockState) {
        return solomonMetricRegistry.createTimer(urlCheckDataBlocksTimerConfiguration,
                SolomonKey.create(SolomonSensor.LABEL_CATEGORY, "urlcheck")
                        .withLabel("block_type", blockType.name())
                        .withLabel("fetch_state", blockState.name())
        );
    }

    private SolomonTimer getCachedTimer(UrlCheckDataBlockType blockType, UrlCheckDataBlockState blockState) {
        return timers.computeIfAbsent(Pair.of(blockType, blockState), igm -> createTimer(blockType, blockState));
    }

    private void updateMetrics(UUID requestId, UrlCheckDataBlockType blockType, UrlCheckDataBlockState blockState) {
        long now = System.currentTimeMillis();
        long start = UUIDs.unixTimestamp(requestId);
        getCachedTimer(blockType, blockState).update(Duration.millis(now - start));
    }
}
