package ru.yandex.webmaster3.storage.user.dao;

import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.core.type.TypeReference;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.notification.LanguageEnum;
import ru.yandex.webmaster3.core.util.enums.EnumResolver;
import ru.yandex.webmaster3.storage.user.ServiceAnnouncement;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.ValueDataMapper;

@Repository("serviceAnnouncementYDao")
public class ServiceAnnouncementYDao extends AbstractYDao {
    private static final DataMapper<ServiceAnnouncement> MAPPER = DataMapper.create(
        F.ROUTE_NAME_REGEXP, F.TEXTS, F.ENABLED, F.EXPERIMENT, F.TYPE, ServiceAnnouncement::new
    );

    private static final ValueDataMapper<ServiceAnnouncement> VALUE_MAPPER = ValueDataMapper.create2(
        Pair.of(F.ROUTE_NAME_REGEXP, ServiceAnnouncement::getRouteNameRegexp),
        Pair.of(F.TEXTS, ServiceAnnouncement::getTexts),
        Pair.of(F.ENABLED, ServiceAnnouncement::isEnabled),
        Pair.of(F.EXPERIMENT, ServiceAnnouncement::getExperiment)
    );

    public ServiceAnnouncementYDao() {
        super("/webmaster3/internal", "service_announcement");
    }

    public void insert(ServiceAnnouncement info) {
        upsert(F.ROUTE_NAME_REGEXP.value(info.getRouteNameRegexp()),
                F.TEXTS.value(info.getTexts()),
                F.ENABLED.value(info.isEnabled()),
                F.EXPERIMENT.value(info.getExperiment())
        ).execute();
    }

    public void batchInsert(List<ServiceAnnouncement> items) {
        batchInsert(VALUE_MAPPER, items).execute();
    }

    public List<ServiceAnnouncement> getServiceAnnouncements() {
        return select(MAPPER)
                .queryForList();
    }

    private static class F {
        static final Field<String> ROUTE_NAME_REGEXP = Fields.stringField("route_name_regexp");
        static final Field<Map<LanguageEnum, String>> TEXTS = Fields.jsonField2("texts", new TypeReference<>() {});
        static final Field<Boolean> ENABLED = Fields.boolField("enabled");
        static final Field<String> EXPERIMENT = Fields.stringField("experiment").makeOptional();
        static final Field<ServiceAnnouncement.Type> TYPE
                = Fields.stringEnumField("type", EnumResolver.er(ServiceAnnouncement.Type.class)).makeOptional();
    }
}
