package ru.yandex.webmaster3.storage.user.dao;

import java.util.LinkedList;
import java.util.List;
import java.util.UUID;
import java.util.function.Consumer;

import org.joda.time.Instant;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.storage.user.SpamBanRule;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;

@Repository
public class SpamBanRulesYDao extends AbstractYDao {
    private static final String TABLE_NAME = "spam_ban_rules";

    public SpamBanRulesYDao() {
        super(PREFIX_INTERNAL, TABLE_NAME);
    }

    public List<SpamBanRule> getRules() {
        List<SpamBanRule> rules = new LinkedList<>();
        forEach(rules::add);
        return rules;
    }

    private void forEach(Consumer<SpamBanRule> cons) {
        streamReader(MAPPER, cons);
    }

    public boolean insertRule(SpamBanRule rule) {
        if (findRule(rule) == null) {
            upsert(F.RULE_ID.value(rule.getRuleId()),
                    F.MASK.value(rule.getMask()),
                    F.SUBNET_SIZE.value(rule.getSubnetLength()),
                    F.OWNER_ID.value(rule.getOwnerId()),
                    F.ACTIVE_STATUS.value(rule.getActiveStatus()),
                    F.CAUSE.value(rule.getCause()),
                    F.UPDATE_DATE.value(rule.getUpdateDate())
            ).execute();
            return true;
        }
        return false;
    }

    public void deleteRule(UUID ruleId) {
        delete()
                .where(F.RULE_ID.eq(ruleId))
                .execute();
    }

    public void updateRule(UUID ruleId, boolean activeStatus, Instant update_date, long adminUserId) {
        update().with(F.ACTIVE_STATUS.set(activeStatus))
                .and(F.UPDATE_DATE.set(update_date))
                .and(F.OWNER_ID.set(adminUserId))
                .where(F.RULE_ID.eq(ruleId))
                .execute();
    }

    private SpamBanRule findRule(SpamBanRule rule) {
        return getRules().stream()
                .filter(item -> item.getMask().equals(rule.getMask()) && (item.getSubnetLength() == rule.getSubnetLength()))
                .findFirst()
                .orElse(null);
    }

    private static final DataMapper<SpamBanRule> MAPPER = DataMapper.create(
            F.RULE_ID, F.MASK, F.SUBNET_SIZE, F.OWNER_ID, F.ACTIVE_STATUS, F.CAUSE, F.UPDATE_DATE, SpamBanRule::new
    );

    private static class F {
        static final Field<UUID> RULE_ID = Fields.uuidField("rule_id");
        static final Field<String> MASK = Fields.stringField("mask");
        static final Field<Integer> SUBNET_SIZE = Fields.intField("subnet_size");
        static final Field<Long> OWNER_ID = Fields.longField("owner_id");
        static final Field<Boolean> ACTIVE_STATUS = Fields.boolField("active_status");
        static final Field<String> CAUSE = Fields.stringField("cause").makeOptional();
        static final Field<Instant> UPDATE_DATE = Fields.jodaInstantField("update_date");
    }
}
