package ru.yandex.webmaster3.storage.user.dao;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.storage.user.SurveyState;
import ru.yandex.webmaster3.storage.user.SurveyStateEnum;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;

@Repository("surveyStateForUserYDao")
@Slf4j
public class SurveyStateYDao extends AbstractYDao {
    private static final DataMapper<SurveyState> MAPPER = DataMapper.create(
            F.TYPE, F.USER_ID, F.HOST_ID, F.STATE, F.DATE, SurveyState::new
    );

    public SurveyStateYDao() {
        super(PREFIX_INTERNAL, "service_survey_state");
    }

    public void insert(SurveyState surveyState) {
        upsert(F.TYPE.value(surveyState.getType()),
                F.USER_ID.value(surveyState.getUserId()),
                F.HOST_ID.value(surveyState.getHostId()),
                F.STATE.value(surveyState.getState()),
                F.DATE.value(surveyState.getDate())
        ).execute();
    }

    public SurveyState getState(String type, long userId, WebmasterHostId hostId, boolean byHost) {
        var select = select(MAPPER)
                .where(F.TYPE.eq(type));
        if (byHost) {
            if (hostId == null) {
                return null;
            }
            select = select
                    .and(F.HOST_ID.eq(hostId));

        } else {
            select = select
                    .and(F.USER_ID.eq(userId));
        }
        return select
                .queryOne();
    }

    public void deleteForUser(long userId) {
        var sel = select(PK_MAPPER).secondaryIndex(USER_ID_INDEX).where(F.USER_ID.eq(userId)).getStatement();
        delete().on(sel).execute();
    }

    private static final DataMapper<Pair<String, Long>> PK_MAPPER = DataMapper.create(
            F.TYPE, F.USER_ID, Pair::of
    );

    private static class F {
        static final Field<String> TYPE = Fields.stringField("type");
        static final Field<Long> USER_ID = Fields.longField("user_id");
        static final Field<WebmasterHostId> HOST_ID = Fields.hostIdField("host_id").makeOptional();
        static final Field<SurveyStateEnum> STATE = Fields.intEnumField("state", SurveyStateEnum.R);
        static final Field<DateTime> DATE = Fields.jodaDateTimeField("update_date");
    }
}
