package ru.yandex.webmaster3.storage.user.dao;

import java.util.List;
import java.util.stream.Collectors;

import lombok.AllArgsConstructor;
import lombok.Value;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.Instant;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.storage.user.message.MessageId;
import ru.yandex.webmaster3.storage.user.message.MessageUpdates;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.ValueDataMapper;

@Repository
public class UserMessageUpdatesYDao extends AbstractYDao {
    public static final String TABLE_NAME = "user_message_updates";

    public UserMessageUpdatesYDao() {
        super(PREFIX_USER, TABLE_NAME);
    }

    public void batchMarkRead(long userId, List<MessageId> messages, boolean read) {
        if (messages.size() == 0) {
            return;
        }
        List<Record> batch =
                messages.stream().map(x -> new Record(userId, x, read, false, Instant.now())).collect(Collectors.toList());
        batchInsert(VALUE_MAPPER, batch).execute();
    }

    public void batchMarkDeleted(long userId, List<MessageId> messages) {
        if (messages.size() == 0) {
            return;
        }
        List<Record> batch =
                messages.stream().map(x -> new Record(userId, x, null, true, Instant.now())).collect(Collectors.toList());
        batchInsert(VALUE_MAPPER, batch).execute();
    }

    public void cleanReadUpdatesForUser(long userId) {
        delete()
                .where(F.USER_ID.eq(userId))
                .and(F.DELETED.eq(false))
                .execute();
    }

    public List<MessageUpdates> getUpdatesForUser(long userId) {
        return select(MAPPER)
                .where(F.USER_ID.eq(userId))
                .queryForList(Pair.of(F.MESSAGE_ID, x -> x.getMessageId().toString()));
    }

    public MessageUpdates getUpdatesForMessage(long userId, MessageId messageId) {
        return select(MAPPER)
                .where(F.USER_ID.eq(userId))
                .and(F.MESSAGE_ID.eq(messageId.toString()))
                .queryOne();
    }

    public void deleteForUser(long userId) {
        delete().where(F.USER_ID.eq(userId)).execute();
    }

    private static final DataMapper<MessageUpdates> MAPPER = DataMapper.create(
            F.MESSAGE_ID, F.READ, F.DELETED,
            (mId, read, deleted) -> new MessageUpdates(MessageId.fromString(mId), read, deleted)
    );

    private static final ValueDataMapper<Record> VALUE_MAPPER = ValueDataMapper.create2(
            Pair.of(F.USER_ID, Record::getUserId),
            Pair.of(F.MESSAGE_ID, x -> x.getMessageId().toString()),
            Pair.of(F.READ, Record::getRead),
            Pair.of(F.DELETED, Record::getDeleted),
            Pair.of(F.UPDATE_TIME, Record::getUpdateTime)
    );

    @Value
    @AllArgsConstructor
    private static class Record {
        Long userId;
        MessageId messageId;
        Boolean read;
        Boolean deleted;
        Instant updateTime;
    }

    private static class F {
        static final Field<Long> USER_ID = Fields.longField("user_id");
        static final Field<String> MESSAGE_ID = Fields.stringField("message_id");
        static final Field<Boolean> READ = Fields.boolField("read").makeOptional();
        static final Field<Boolean> DELETED = Fields.boolField("deleted").withDefault(false).makeOptional();
        static final Field<Instant> UPDATE_TIME = Fields.jodaInstantField("update_time");
    }
}
